## Sid Meier's Civilization 4
#
# CvAsciiTable.py
# This file contains a class which allows one to create ASCII tables out of
# 2D arrays.
#
# This file is part of ClipboardDemographicsMod by Vovan
#

class CvAsciiTable:
	def getText(self, data):
		"""
			data - Array of arrays
			
			Returns an ASCII table representation of the data
		"""
		columnWidths = self.calculateColumnWidths(data, self.countColumns(data))
		line_list = []
		
		separator = self.makeSeparator(columnWidths)
		
		for row in data:
			line_list.append(separator)
			line_list.append("\r\n")
			line_list.extend(self.makeRow(row, columnWidths))
			line_list.append("\r\n")
		
		line_list.append(separator)
		
		return ''.join(line_list)
	
	def countColumns(self, data):
		" Returns the number of columns in the data "
		result = 0
		for row in data:
			result = max(result, len(row))
		return result
	
	def calculateColumnWidths(self, data, columnCount):
		"Returns a list of integers that denote the width of each column in characters"
		result = []
		for i in range(columnCount):
			result.append(self.calculateColumnWidth(data, i))
		return result
	
	def calculateColumnWidth(self, data, col):
		"Returns the width of the 'col'th column of the data in characters"
		width = 0
		for row in data:
			cols = len(row)
			if col in range(cols):
				width = max(width, len(unicode(row[col])))
		return width
	
	def makeSeparator(self, columnWidths):
		" Given a list of column widths, makes an ASCII separator thingey to separate rows of the table. "
		from array import array
		chars = array('c')
		chars.append('+')
		for w in columnWidths:
			for i in range(w):
				chars.append('-')
			chars.append('+')
		return chars.tostring()
	
	def makeRow(self, row, columnWidths):
		" Makes an ASCII representation of the row passed in. Returns a list of strings to append. "
		str_list = ["|"]
		colCount = len(columnWidths)
		actualColCount = len(row)
		for i in range(actualColCount):
			str_list.append(self.makeCell(unicode(row[i]), columnWidths[i]))
		for i in range(actualColCount, colCount):
			str_list.append(self.makeCell("", columnWidths[i]))
		return str_list
	
	def makeCell(self, content, width):
		" Makes a single cell with the specified width and contents. "
		from array import array
		chars = array('u')
		widthDiff = width - len(content)
		left = widthDiff / 2
		right = left
		if (widthDiff % 2) > 0:
			right = right + 1
		for i in range(left):
			chars.append(u' ')
		chars.fromunicode(content)
		for i in range(right):
			chars.append(u' ')
		chars.append(u'|')
		return str(chars.tounicode().encode('ascii', 'replace'))
	