#
#	FILE:	 Fair_Continents.py
#	VERSION: 2.0
#	AUTHOR:  Doug McCreary (surt@uga.edu)
#	PURPOSE: Create some pretty fair continents
#            and demonstrate non-fractal map generation
#-----------------------------------------------------------------------------
#	Copyright (c) 2005-2006 Doug's Software All rights reserved.
#-----------------------------------------------------------------------------
#

from CvPythonExtensions import *
import CvUtil
import CvMapGeneratorUtil
from CvMapGeneratorUtil import FractalWorld
from CvMapGeneratorUtil import TerrainGenerator
from CvMapGeneratorUtil import FeatureGenerator

def getDescription():
	return unicode("Fair Continents")

#2 custom options, number of continents horizontal and vertical
def getNumCustomMapOptions():
	return 2
	
#horizontal and vertical option titles for the setup screen
def getCustomMapOptionName(argsList):
	print "getCustomMapOptionName",argsList
	index = argsList[0]
	translated_text = [unicode("Horizontal"),
	                   unicode("Vertical")]
	return translated_text[index]
	
#there are 4 possible values for each
def getNumCustomMapOptionValues(argsList):
	print "getNumCustomMapOptionValues",argsList
	index = argsList[0]
	typeCount = [4,4]
	return typeCount[index]
	
#here are the 4 options for each, from 2 to 5 wide/tall	
def getCustomMapOptionDescAt(argsList):
	print "getCustomMapOptionDescAt",argsList
	iOption = argsList[0]
	iSelection = argsList[1]
	selection_names = [
	                  ["2 wide",
	                   "3 wide",
	                   "4 wide",
	                   "5 wide"],
	                  ["2 tall",
	                   "3 tall",
	                   "4 tall",
	                   "5 tall"]]
	translated_text = unicode(selection_names[iOption][iSelection])
	return translated_text
	
#defaults to 3 wide by 3 tall	
def getCustomMapOptionDefault(argsList):
	return 1

#this is an advanced map
def isAdvancedMap():
	"This map should not show up in simple mode"
	return 1

#this map uses the 'sea level' option to determine how much ocean
#goes between the continents (return 0 to disable the sea level selector)
def isSeaLevelMap():
	return 1

#starts the player's warrior and settler on the same tile if true
def startHumansOnSameTile():
	return True
	
#allows you to override the grid sizes if your map has some particular need
#in my case, i'm going to force all the grid sizes to be odd so that the 
#ocean stripes will be more fair to the central players	
def getGridSize(argsList):
	# Make grid sizes all odd, and also approx twice as wide as tall so that
	# horizontal and vertical travel are fair.
	# the longest horizontal trip is 1/2 width due to being able to navigate
	# in either direction.  the longest vertical trip is 1 *height because
	# height doesn't wrap
	grid_sizes = {
		WorldSizeTypes.WORLDSIZE_DUEL:		(11,5),
		WorldSizeTypes.WORLDSIZE_TINY:		(15,7),
		WorldSizeTypes.WORLDSIZE_SMALL:		(19,9),
		WorldSizeTypes.WORLDSIZE_STANDARD:	(23,11),
		WorldSizeTypes.WORLDSIZE_LARGE:		(27,13),
		WorldSizeTypes.WORLDSIZE_HUGE:		(35,17)
	}

	if (argsList[0] == -1): # (-1,) is passed to function on loads
		return []
	[eWorldSize] = argsList
	return grid_sizes[eWorldSize]
	

#first step: lay out the plot types, which is a sort of broad view of the map tiles
#basically telling you if you are looking at ocean, land, hill, or mountain
def generatePlotTypes():
	NiTextOut("Setting Plot Types (Python Fair Continents) ...")
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	plotTypes = [PlotTypes.PLOT_OCEAN] * (iW*iH)
	for plotX in range(iW):
		for plotY in range(iH):
			index = (plotY*iW) + plotX
			if (plotY <= 1 or plotY >= iH-2):
				plotTypes[index] = PlotTypes.PLOT_OCEAN
			else:
				plotTypes[index] = PlotTypes.PLOT_LAND
				
	# Sea Level adjustment (from user input), transformed into width of sea grids
	sea = gc.getSeaLevelInfo(map.getSeaLevel()).getSeaLevelChange()
	seaWidth = 2 #default width for ocean stripes for normal sea level
	if (sea > 0):
		seaWidth = 3 #high sea level means extra width on the ocean stripes
	if (sea < 0):
		seaWidth = 1 #low sea level mean little width on the ocean stripes
		
	print "seawidth",seaWidth
				
				
	#add ocean stripes to border continents
	userInputNumVerStripes = map.getCustomMapOption(0) + 1
	userInputNumHorStripes = map.getCustomMapOption(1) + 1
	print "userInputNumVerStripes",userInputNumVerStripes
	print "userInputNumHorStripes",userInputNumHorStripes
	for stripe in range(1,userInputNumVerStripes+1):
		print "stripe",stripe
		verStripePos = int( (iW*stripe) / (userInputNumVerStripes+1))
		print "verStripePos",verStripePos
		for xpos in range(verStripePos-seaWidth, verStripePos+seaWidth+1):
			for ypos in range(iH):
				index = (ypos*iW) + xpos
				print "oceanver x", xpos, "y", ypos
				plotTypes[index] = PlotTypes.PLOT_OCEAN
	for stripe in range(1,userInputNumHorStripes+1):
		horStripePos = int( (iH*stripe) / (userInputNumHorStripes+1))
		print "horStripePos",horStripePos
		for xpos in range(iW):
			for ypos in range(horStripePos-seaWidth, horStripePos+seaWidth+1):
				index = (ypos*iW) + xpos
				print "oceanhor x", xpos, "y", ypos
				plotTypes[index] = PlotTypes.PLOT_OCEAN
		

	#Extra vertical stripe to prevent horizontal continent wrapping
	#placed on the horizontal border wrap
	#sized to match the sea level setting
	borderStripePositions = [0,1,2,(iW-2),(iW-1)]
	if (seaWidth == 3 ):
		borderStripePositions = [0,1,2,3,(iW-3),(iW-2),(iW-1)]
	if (seaWidth == 1 ):
		borderStripePositions = [0,(iW-2),(iW-1)]
		
	
	print "extra verstripe"
	for xpos in borderStripePositions:
		for ypos in range(iH):
			index = (ypos*iW) + xpos
			print "oceanedge x", xpos, "y", ypos
			plotTypes[index] = PlotTypes.PLOT_OCEAN

	# Add some random ocean to unsquarify the continents
	#directions identify the eight neighbor tiles
	directions = ( (-1,-1), (-1,0), (0,-1), (-1,1), (1,-1), (1,0), (0,1), (1,1) )
	iExtraOcean = int((iW*iH) / 2)
	while iExtraOcean > 0:
		iExtraOcean = iExtraOcean - 1
		oceanX = 1 + dice.get(iW-3, "Random Ocean x - Fair Continents")
		oceanY = 1 + dice.get(iH-3, "Random Ocean y - Fair Continents")
		index = (oceanY*iW) + oceanX
		#check for a nearby ocean, this makes sure every added ocean touches the main ocean
		#note that we don't have to worry about going off the board because we've restricted
		#the search to positions at least one indent inside the board
		for direction in directions:
			xOffset, yOffset = direction
			nearIndex = ( ((oceanY+yOffset)*iW) + (oceanX + xOffset))
			if (plotTypes[nearIndex] == PlotTypes.PLOT_OCEAN):
				plotTypes[index] = PlotTypes.PLOT_OCEAN
				print "oceantouchextra x", oceanX, "y", oceanY

	# Add some random lakes to make terrain more interesting
	iExtraOcean = int((iW*iH) / 25)
	while iExtraOcean > 0:
		iExtraOcean = iExtraOcean - 1
		oceanX = dice.get(iW, "Random Ocean x - Fair Continents")
		oceanY = dice.get(iH, "Random Ocean y - Fair Continents")
		index = (oceanY*iW) + oceanX
		print "oceanextra x", oceanX, "y", oceanY
		plotTypes[index] = PlotTypes.PLOT_OCEAN

	# Add some random hills to make terrain more interesting
	#note this isn't quite 1/7 = 14% hills, because it only hills if the random
	#draw hits a land tile
	iExtraHills = int((iW*iH) / 7)
	while iExtraHills > 0:
		iExtraHills = iExtraHills - 1
		hillX = dice.get(iW, "Random Hill x - Fair Continents")
		hillY = dice.get(iH, "Random Hill y - Fair Continents")
		index = (hillY*iW) + hillX
		print "hillextra x", hillX, "y", hillY
		if plotTypes[index] == PlotTypes.PLOT_LAND:
			plotTypes[index] = PlotTypes.PLOT_HILLS

	# Add some random peaks to make terrain more interesting
	#note that this isn't quite adding 1/15 = 6.6% peaks because it only peaks when
	#the random draw is near a hill or an existing peak
	iExtraPeaks = int((iW*iH) / 15)
	while iExtraPeaks > 0:
		iExtraPeaks = iExtraPeaks - 1
		peakX = 1+dice.get(iW-3, "Random Peak x - Fair Continents")
		peakY = 1+dice.get(iH-3, "Random Peak y - Fair Continents")
		index = (peakY*iW) + peakX
		#check for a nearby hill, this makes sure every added peak touches a hill
		#note that we don't have to worry about going off the board because we've restricted
		#the search to positions at least one indent inside the board
		for direction in directions:
			xOffset, yOffset = direction
			nearIndex = ( ((peakY+yOffset)*iW) + (peakX + xOffset))
			if (plotTypes[nearIndex] == PlotTypes.PLOT_HILLS or plotTypes[nearIndex] == PlotTypes.PLOT_PEAK):
				plotTypes[index] = PlotTypes.PLOT_PEAK
				print "peakextra x", peakX, "y", peakY

	# Finished generating plots!
	return plotTypes
	
#now generate the fine terrain, in particular, need to decide if 
#any land tile is desert, plains, grass, tundra, or snow
def generateTerrain():		
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	terrainData = [TerrainTypes.NO_TERRAIN]*(iW*iH)
	
	terrainDesert = gc.getInfoTypeForString("TERRAIN_DESERT")
	terrainPlains = gc.getInfoTypeForString("TERRAIN_PLAINS")
	terrainGrass = gc.getInfoTypeForString("TERRAIN_GRASS")
	terrainTundra = gc.getInfoTypeForString("TERRAIN_TUNDRA")
	terrainSnow = gc.getInfoTypeForString("TERRAIN_SNOW")
	midmap = int(iH / 2)
	midrange = int(iH / 10)
	mindesert = int(midmap - midrange)
	maxdesert = int(midmap + midrange)
	
	for x in range(iW):
		for y in range(iH):
			pPlot = map.plot(x,y)
			index = (y*iW) + x
			plotType = pPlot.getPlotType()
			plotRand = dice.get(100, "Random Terrain - Fair Continents Python")
			#water tiles, peaks, and hills should always take their default terrain type
			if (pPlot.isWater()):
				terrainData[index] = pPlot.getTerrainType()
			elif (plotType == PlotTypes.PLOT_PEAK):	
				terrainData[index] = pPlot.getTerrainType()
			elif (plotType == PlotTypes.PLOT_HILLS):	
				terrainData[index] = pPlot.getTerrainType()
			#land tiles actually have choices in what type of land tile to be
			elif (plotType == PlotTypes.PLOT_LAND):
				#note that iH is at the top of the map (northmost), when you might expect bottom
				#this allows snow only on the very furthest north or south strip of land
				if ( (y < 3 or y > iH-4) and plotRand < 10):
					terrainData[index] = terrainSnow
				#this allows tundra only within 3 tiles of the north/south most land position
				elif ( (y < 5 or y > iH-6) and plotRand < 50):
					terrainData[index] = terrainTundra
				elif (plotRand < 50):
					terrainData[index] = terrainGrass
				elif (plotRand < 90):
					terrainData[index] = terrainPlains
				#desert only allowed within 10% stripe in center of map
				elif ( (y > mindesert) and (y < maxdesert) ):
					terrainData[index] = terrainDesert
				else:
					terrainData[index] = terrainGrass
			else:
				terrainData[index] = pPlot.getTerrainType()
			#this should be impossible, but just in case, make sure a terrain is assigned
			if (terrainData[index] == TerrainTypes.NO_TERRAIN):
				terrainData[index] = pPlot.getTerrainType()
				
	return terrainData
	
#this is the standard interface method, which i'm just rerouting to my generateTerrain method	
def generateTerrainTypes():
	NiTextOut("Generating Terrain (Fair Continents) ...")
	terrainTypes = generateTerrain()
	return terrainTypes
	
#this is responsible for things like oasis, ice floes on the ocean, goodies ...
#haven't figured out how all of this works yet
def addFeatures():
	# Remove all peaks along the coasts, before adding Features, Bonuses, Goodies, etc.
	map = CyMap()
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	for plotIndex in range(iW * iH):
		pPlot = map.plotByIndex(plotIndex)
		if pPlot.isPeak() and pPlot.isCoastalLand():
			# If a peak is along the coast, change to hills and recalc.
			pPlot.setPlotType(PlotTypes.PLOT_HILLS, true, true)
			
	# Now add the features.
	NiTextOut("Adding Features (Python Fair Continents) ...")
	featuregen = FeatureGenerator()
	featuregen.addFeatures()
	featuregen.addFeatures() #try calling an extra time for extra features?
	return 0

#handled in addfeatures, so unnecessary
def normalizeRemovePeaks():
	return None

#these are the resources you find in water tiles
resourcesWater = ('BONUS_CLAM', 'BONUS_FISH', 'BONUS_WHALE', 'BONUS_CRAB')

#this method puts down the tile bonuses, such as whale, rice, oil, etc.	
def addBonusType(argsList):
	[iBonusType] = argsList
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	type_string = gc.getBonusInfo(iBonusType).getType()

	sizekey = map.getWorldSize()
	sizevalues = {
		WorldSizeTypes.WORLDSIZE_DUEL:		1,
		WorldSizeTypes.WORLDSIZE_TINY:		2,
		WorldSizeTypes.WORLDSIZE_SMALL:		4,
		WorldSizeTypes.WORLDSIZE_STANDARD:	7,
		WorldSizeTypes.WORLDSIZE_LARGE:		11,
		WorldSizeTypes.WORLDSIZE_HUGE:		16
		}
	sizemodifier = sizevalues[sizekey]
	#find out how many players in game, and make sure there are some for each
	players = gc.getGame().countCivPlayersEverAlive()
	playerExtra = dice.get(int(players / 2) + 1, "Extra resources per player - Fair Continents Python")
	extraWater = 0
	#the bountiful ocean - I add extra resources to water tiles because there
	#is a lot of coastal land in this map design
	if (type_string in resourcesWater):
		extraWater = players; #extra one of each water bonus per player
	
	#total is based on: size of map, num players, random extra per player, extra for water types
	count = int(sizemodifier + players + playerExtra + extraWater)
	
	# Set plot eligibility for current bonus.
	# Begin by initiating the list, into which eligible plots will be recorded.
	eligible = []
	for x in range(iW):
		for y in range(iH):
			# First check the plot for an existing bonus.
			pPlot = map.plot(x,y)
			if pPlot.getBonusType(-1) != -1:
				continue # to next plot.
			if pPlot.getFeatureType() == gc.getInfoTypeForString("FEATURE_OASIS"):
				continue # Soren wants no bonuses in oasis plots. So mote it be.
			# Check plot type and features for eligibility.
			if (pPlot.canHaveBonus(iBonusType, True)):
				eligible.append([x,y])
                                    
	# Now we assign the bonuses to eligible plots chosen completely at random
	# from among the valid plots found above
	while count > 0:
		if eligible == []: break # No eligible plots left!
		index = dice.get(len(eligible), "Bonus Placement - Fair Continents PYTHON")
		[x,y] = eligible[index]
		map.plot(x,y).setBonusType(iBonusType)
		del eligible[index] # Remove this plot from the eligible list.
		count = count - 1  # Reduce number of bonuses left to place.
	# This bonus type is done.
	
	# Uncomment to Let C also handle this bonus in the default way.
	#CyPythonMgr().allowDefaultImpl() 
