#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

void build_lbx(void);
void extract_lbx(void);
int read_offset(FILE * file);
int compute_offset(int cnt);
void fwrite_offset(FILE * file, int cnt);
void fwrite_file(FILE * file, int cnt, int bytes);
unsigned int extract_bits(unsigned int val, int b1, int b2);


int main(void)
{
	char choice[100];

	choice[0] = '\0';
	while (choice[0] != 'B' && choice[0] != 'E')
	{
	  printf("Build .lbx from a directory or extract an lbx to a directory? (B/E)");
	  scanf("%s", choice);
	  choice[0] = toupper(choice[0]);
	}
	if (choice[0] == 'B')
		build_lbx();
	else
		extract_lbx();


	printf("Exiting Tool.\n");
	return 0;
}

void extract_lbx(void)
{
	FILE * file;
	char filename[100];
	char outfilename[100];
	int n_files;
	FILE * outfile;
	int n1, n2;
	int byte_count = 0;
	int offsets[1000];
	int cnt;

	printf("Enter Filename of the lbx archive: ");
	scanf("%s", filename);
	file = fopen(filename, "rb");
	if (file == NULL)
	{
		printf("Error opening file.\n");
		return;
	}

	n1 = fgetc(file);
	if (feof(file)) return;
	n2 = fgetc(file);
	if (feof(file)) return;
	n_files = n2*256+n1;
	byte_count = 2;

	printf("Number of files to extract: %d\n", n_files);
	
	/* Read past LBX File Identifier & version # */
	fgetc(file);
	fgetc(file);
	fgetc(file);
	fgetc(file);  // End of LBX identifier = AD FE 00 00
	fgetc(file); 
	fgetc(file);  // End of file version 00 00
	byte_count += 6;

	/* One extra offset for EOF */
	for (cnt = 0; cnt <= n_files; cnt++)
	{
		offsets[cnt] = read_offset(file);
		if (feof(file))
		{
			printf("Error: Unexpected End of File\n");
			return;
		}
		byte_count += 4;
		printf("Offset: File #%d = %d\n", cnt, offsets[cnt]);
	}

	/* Read to first offset */
	while (byte_count < offsets[0])
	{
		fgetc(file);
		byte_count++;
		if (feof(file))
		{
			printf("Error: Unexpected End of File\n");
			return;
		}
	}

	/* Extract the files */
	cnt = 0;
	while (cnt < n_files)
	{
		sprintf(outfilename, "file%d.txt", cnt+1);
		outfile = fopen(outfilename, "wb");
		if (outfile == NULL)
		{
			printf("Error opening output file %s\n", outfilename);
			return;
		}
		while (byte_count < offsets[cnt+1])
		{
			fprintf(outfile, "%c", fgetc(file));
			byte_count++;
			if (feof(file))
			{
				printf("Error: Unexpected End of File\n");
				return;
			}
		}
		fclose(outfile);
		cnt++;
	}

	fclose(file);
	return;
}

int read_offset(FILE * file)
{
	int n1, n2, n3, n4;

	n1 = fgetc(file);
	if (feof(file)) return 0;
	n2 = fgetc(file);
	if (feof(file)) return 0;
	n3 = fgetc(file);
	if (feof(file)) return 0;
	n4 = fgetc(file);
	if (feof(file)) return 0;


	return (65536*n4 + 4096*n3 + 256*n2 + n1);
}

void build_lbx(void)
{
	char filename[100];
	FILE * file;
	int n_files;
	unsigned int offsets[1000];
	int cnt;
	int n1, n2;
	int n_bytes;

	printf("Enter LBX archive name to build: \n");
	scanf("%s", filename);
	printf("# of files to include: ");
	scanf("%d", &n_files);
	if (n_files < 1 || n_files > 1000)
	{
		printf("# of files must be between 1 and 1000\n");
		return;
	}	
	file = fopen(filename, "wb");
	if (file == NULL)
	{
		printf("Error opening %s for output.\n", filename);
		return;
	}

	/* Compute our offsets */
	/* Start of 1st file */
	/*  This wasn't working trying to always use 2048 like some of the files ive seen 
	offsets[0] = 8 + (n_files+1)*4; // 8 bytes header + 4 bytes per offset
	*/
	offsets[0] = 2048;

	cnt = 1;
	while (cnt <= n_files)
	{
		offsets[cnt] = offsets[cnt-1] + compute_offset(cnt); // Start of previous offset
		cnt++;
	}

	/* Write header */
	n1 = n_files & 255;
	n2 = n_files & 65280;
	n2 = n2 >> 8;

	fprintf(file, "%c%c%c%c%c%c%c%c", n1, n2, 173, 254, 0, 0, 0, 0);

	/* Write out our n+1 offsets */
	cnt = 0;
	while (cnt <= n_files)
	{
		fwrite_offset(file, offsets[cnt]);
		cnt++;
	}
	
	/* write filler up to first offset */
	n_bytes = 8 + (n_files+1)*4;
	while ((unsigned) n_bytes < offsets[0])
	{
		fprintf(file, "%c", 0);
		n_bytes++;
	}

	/* now write out each of the files */
	cnt = 0;
	while (cnt < n_files)
	{
		n_bytes = offsets[cnt+1] - offsets[cnt];
		fwrite_file(file,  cnt, n_bytes);
		cnt++;
	}
	fclose(file);
	return;
}

void fwrite_file(FILE * file, int cnt, int bytes)
{
	FILE * outfile;
	char filename[100];
	int cnt2;

	sprintf(filename, "file%d.txt", (cnt+1));
	outfile = fopen(filename, "rb");
	if (outfile == NULL)
	{
		printf("Error opening file: %s\n", filename);
		exit (-1);
	}
	
	printf("Archiving: %s  (%d bytes)\n", filename, bytes);
	cnt2 = 0;
	while (cnt2 < bytes)
	{
		fprintf(file, "%c", fgetc(outfile));
		if (feof(outfile))
		{
			printf("Error:  Unexpected end of file.\n");
			exit (-1);
		}
		cnt2++;
	}
	fclose(outfile);
	return;
}

void fwrite_offset(FILE * file, int offset)
{
	int n1, n2, n3, n4;

	n1 = offset & 255;
	n2 = offset & 65280;
	n3 = offset & 116711680;
    n4 = offset & 4278190080;

	n2 = n2 >> 8;
	n3 = n3 >> 16;
	n4 = n4 >> 24;

	fprintf(file, "%c%c%c%c", n1, n2, n3, n4);
	return;	
}


int compute_offset(int cnt)
{
	char filename[100];
	int byte_count = 0;
	FILE * file;

	sprintf(filename, "file%d.txt", cnt);
	file = fopen(filename, "r");
	if (file == NULL)
	{
		printf("Error opening file %s\n", filename);
		exit(-1);
	}
	while (!feof(file))
	{
		fgetc(file);
		if (!feof(file))
			byte_count++;
	}
	printf("Size of file #%d = %d\n", cnt, byte_count);

	fclose(file);
	return byte_count;
}