#
#	FILE:	 SmartMap.py
#	VERSION: 2.0
#	AUTHOR:  Doug McCreary (surt@uga.edu)
#	PURPOSE: Allow a finely detailed user control of map generation
#-----------------------------------------------------------------------------
#	Copyright (c) 2005-2006 Doug's Software All rights reserved.
#-----------------------------------------------------------------------------
#

# Version History
# 7.0
#     Add new generation option: Land Style.  Land style allows you to opt
#       for less roundish continents, and also to allow for small independent
#       land fragments separate from the main continents.  When land fragments
#       are generated, they may be generated anywhere, and may violate the 
#       sea level strategy guarantees, so use at your own risk.  The less rounded
#       continent options without land fragments will obey the sea level rules,
#       but may be less fair.
#     Based on a report of it being annoying to have your capital placed on
#       the wrap-edge of the minimap, I now rotate the initial land placement
#       to put the minimum number possible of land near the map wrap edge.
#       While this does not guarantee this will not happen, it makes it highly
#       unlikely.
#     Further, made the first player always pick the center most start position
#       if wrap is enabled.  This should make single player games always have
#       the human player start fairly well centered.  This should absolutely
#       guarantee that a single human player will not start with their capital
#       on a map wrap edge.
#     Fixed a couple of places where I did not consider y-wrapping correctly.
#       But I guess no one plays y-wrap (or both wrap), or they'd have surely noticed.
#     Introduce option 'SmartMap Strict' for feature generation.  This will only place
#       floodplains & tundra where the normal scripts would.
#     Made floodplains on anything other than desert less likely, and oasis on tundra
#       less likely for the non-strict feature generator.
#     Enhanced the plot wetness calculator so plot wetness can be factored into more
#       of the generation process.
#     Improved the player placement rules to prefer not to place players initially
#       on a stretch of land reaching into the ocean.  
#     Added ReadMeSmartMap.txt
# 6.1
#     Tweak up number of hills for 'many hills'.
# 6.0
#     Improved lake generation system, grows more lakes beyond one plot
#     Numerous terrain relevance improvements.  More desert&snow at high altitudes,
#       less desert near water, etc.
#     Changed continent computation to more accurately determine the large
#       continents, which should as a side effect improve the fair resource
#       distribution.  Removed tiny continents which will never be player
#       starts from the fair resource distribution.
#     Improved tile wetness calculator to consider number of rivers and lakes
#       touched by tile (a tile touched by fresh water on multiple sides is
#       now more likely to shift from desert->plains->grass).
#     Added a river originating at the highest point of each continent, which
#       greatly improves the fairness of river distribution overall
#     Changed forest/jungle options to allow independent choice of forest and
#       jungle coverage (light,normal,heavy independently selectable for each)
#     General code cleanup, reorganization, commenting
#     Ran code through pychecker and cleaned all warnings, pychecker found a
#       couple of legitimate bugs that may have improved the resulting maps.
# 5.1
#     Improvements to fair bonus distribution algorithm.
#     Fixed a typo bug causing python error on undefined edgeprob
#     Tweak desert rate down slightly
# 5.0
#     Work on refining the continent boundaries so water straights are not so
#       obvious when you see the resulting cymap.
#     Rewrite river generation, move it before terrain generation so terrains
#       can consider rivers (allow grasslands and other 'wetter' terrain to be
#       more common around rivers, as well as forests/jungle)
#     Replace the standard river altitude function with one that seems to yeild
#       better river paths
#     Assorted terrain and feature improvements.  Restored possibility of flood
#       plains in desert, and made terrain tend to blob together more rather
#       than appearing so completely random.  Allowed hills to have terrain rather
#       than always defaulting to grass (so desert/hills are now possible).  I can't
#       believe no one complained about that one.
#     Found a major logic error in the player placement algorithm that was unfairly
#       favoring near-lake placements (I has assumed CyPlot.isCoastalLand() was equivalent
#       to asking: can I build a lighthouse?, this turned out not to be true).
#       As a result, initial placement is much more likely to be coastal now, and 
#       this has a nice side effect of improving the average separation of players.
#     Tweaked resource rates a little more based on various feedback.
#     Replace standard lake addition function so we don't get lakes in odd places
#     Futher performance optimization.  The low and medium sea level options are
#       now more than twice as fast, and the high sea level option is about 50%
#       faster.  I can now generate a huge map with low sea level in under a minute.
#     After discovering that the map sizes in great plains are way off, resized my 
#       maps to the following settings:
#                   Duel Tiny Small Standard Large Huge
#         SmartMap: 60   100  180   320      560   1000
#         Standard: 60   104  160   273      416   640
#       This is a bump up for everything except huge, which becomes slightly smaller,
#       and thus hopefully usable by more people.
#     Added related option to override the map width and height.  You can now select
#       your map size fairly precisely, including the ability to pick map sizes that
#       are so large they may cause civiv to have problems.  Up to 256x256 map positions,
#       which would be a 1024x1024 map, with an area of over 1 million (compare to huge at
#       an area of 16 thousand!)  If you are able to generate a map of even 128x128 I'll be
#       shocked.  Nevertheless, the option is there.  So no complaining that I made huge
#       smaller!  Just set your game to huge, then override the size to whatever you want.
#       Or for a bizarre experience, set your map to duel, and override the size to whatever
#       you'd like.  For kicks I generated a 256x8 map, and that was just barely startable
#       with my 1 Gig of memory.  Minimap looks ridiculous.
# 4.2
#     Increase goody hut count slightly, particularly on maps with lots of land
# 4.1
#     Slightly refined smartmap resource placement to make distribution
#       fairer between continents, and not to allow so many duped resources
#       within one city radius
#     Small upward tweak to resource rates
# 4.0
#     Overall performance improvement:
#        Now down to about: 
#        <5 seconds for duel,tiny,small
#        10 seconds for standard,
#        40 seconds for large,
#        ~2 minutes on huge
#        On a 1.8Ghz Pentium-M with 1Gig
#     Optimize land placement speed by remembering all potential growth positions
#       rather than looking for them randomly.
#     Optimized and improved player placement function.  No longer makes certain
#       rare placement errors, and the speed is dramatically better, particularly
#       for continents with a large number of players to be placed (pangea).
#     Optimized bonus placement.  This used to get very slow on large/huge maps due
#       to making a pass through all tiles for each bonus type.  Now makes one pass
#       through all tiles and caches the result.
#     Changed the player option read function to take a string rather than an index
#       to make it clearer in code, and also make it impossible to break when
#       changing the order of options in the list.
#     Land areas are 20,40,90,210,500,1200 (duel,tiny,small,standard,large,huge)
#       Compared to  15,24,48,88 ,154,252 for great plains.
#       Smartmap rounds down calculated width and height,
#       so the areas are not quite as big as they seem,
#       but large and huge are noticeably bigger.  This allows for more flexible
#       use of the ocean coverage option.  Note to firaxis: it would be a better
#       design if in the expansion pack or a patch you passed the expected land
#       area to the grid size function from the current mod!  Then maps could
#       dynamically size themselves to the expected land area and provide fair
#       gameplay regardless of other chosen settings (and be compatible with mods
#       that make use of different sized maps ... no more hardcoding like in
#       great plains)!
#       Note: I can just barely start a huge game with 1Gig memory.  This implies
#       firaxis is using somewhere in the neighborhood of 32k memory per tile.
#       Good grief, what on earth are they storing!
# 3.2
#     Add options for specifying weight of initial jungle/forest cover
#     More performance optimizations
# 3.1 
#     Fix rare issue with placements landing too close together
#     Slight tweaks to placement scoring to value early resources higher
#     Fix issue with placements on separate continents too close together
# 3.0 
#     Re-enable climate selection, which may or may not be honored
#     Restore 36-continent option, accidental deletion
#     Create a better default option for each selector
#     Un-smooth edges near poles
#     Properly handle allowing continents to wrap on wrap axis
#     Performance optimizations by reducing function call counts
#     Custom player placement rules
#     Complete replacement of bonus system
#     Separate out goody hut placement and put in correct order
#     Cleaner debug print messages
# 2.0 
#     Reduce bonus percentage for SmartMap bonus
#     Allow standard bonus distribution as an option
#     Bias width,height towards wrap direction
#     un-straight-edge where continents meet
#     SmartMap terrainGen allows more desert
#     SmartMap featureGen suggests more forest and jungle
#     Support for random ranges in number of continents
# 1.0 
#     Initial version

##########################################################################
#the imports required for SmartMap or the borrowed fractal generators
import operator
from CvPythonExtensions import *
import CvUtil
import CvMapGeneratorUtil
from CvMapGeneratorUtil import FractalWorld
from CvMapGeneratorUtil import TerrainGenerator
from CvMapGeneratorUtil import FeatureGenerator

##########################################################################
#global variables	
wrapX = False #is map wrapped on x
wrapY = False #is map wrapped on y
width = 0 #width of the map in plots
height = 0 #height of the map in plots
#the eight tiles that immediately surround a given tile
cardinalDirs = [(-1,0),(0,-1),(0,1),(1,0)]	
#the eight tiles that immediately surround a given tile
nearDirs = [(-1,-1),(-1,0),(-1,1),(0,-1),(0,1),(1,-1),(1,0),(1,1)]	
#the set of tiles that can be used by a given tile if a city is built there
playerStartCross = [(-1,-1),(-1,0),(-1,1),(0,-1),(0,1),(1,-1),(1,0),(1,1),
                    (-2,-1),(-2,0),(-2,1),(2,-1),(2,0),(2,1),(-1,2),(0,2),(1,2),(-1,-2),(0,-2),(1,-2)]

#a list of lists: of the continents, and all their land tiles
postContinentLists = []	
#remember how many total land tiles we have
totalLandCount = 0
#pre-computed altitudes of all plots
preComputedAltitudes = []
#which map size was selected
sizeSelected = 0
#pre-computed wetness of all plots
preComputedWetness = []


##########################################################################
#Step 1: the map grid size is determined. your map will contain 4 tiles
# for every grid size, so an 8x4 grid size is 32x16 tiles
def getGridSize(argsList):
	if (argsList[0] == -1): # (-1,) is passed to function on loads
		return (-1,-1)
	beforeStepOne()
	OutputMessage("Python: SmartMap: Step 1 Setting Map Size")
	global wrapX
	global wrapY
	gc = CyGlobalContext()
	dice = gc.getGame().getMapRand()
	
	
	#scale the world size to match the area supplied by typical maps
	#this data is (map area, minDimension)
	#each step is 1.8x previous area, rounded down to nearest 20
	#these areas are considerably larger than, for example, great plains
	#which has areas: 15,24,48,88,154,252
	#note that a 60x60 which is bigger than my 1Gig machine can handle 
	#would be 3600, or 3x the area of this huge
	grid_infos = {
		WorldSizeTypes.WORLDSIZE_DUEL:		(60,3),
		WorldSizeTypes.WORLDSIZE_TINY:		(100,4),
		WorldSizeTypes.WORLDSIZE_SMALL:		(180,6),
		WorldSizeTypes.WORLDSIZE_STANDARD:	(320,10),
		WorldSizeTypes.WORLDSIZE_LARGE:		(560,16),
		WorldSizeTypes.WORLDSIZE_HUGE:		(1000,24)
	}
	
	
	#extract the right world size
	[eWorldSize] = argsList
	print "    [eWorldSize]",eWorldSize	
	mapArea, minDimension = grid_infos[eWorldSize]
	print "    mapArea",mapArea	
	
	global sizeSelected
	sizeSelected = eWorldSize

	
	#calculate a random width that will fit the world size
	maxDimension = mapArea / minDimension
	print "    minDimension",minDimension	
	print "    maxDimension",maxDimension	
	calcwidth = minDimension + dice.get(1+(maxDimension-minDimension), "Python: SmartMap: Dice: Random Continent X")
	#bias the width or height toward the wrap direction if the map wraps in only one direction
	if wrapX and not wrapY:
		calcwidth += int(calcwidth/5)
	if not wrapX and wrapY:
		calcwidth -= int(calcwidth/5)
	if calcwidth > 50:
		calcwidth = 50
	if calcwidth < 6:
		calcwidth = 6
	
	calcheight = int(mapArea / calcwidth) + 1
	
	if calcheight > 60:
		calcheight = 60
	if calcheight < 6:
		calcheight = 6
		
	#make sure the map isn't too big	
	calculatedArea = calcwidth * calcheight
	while (calculatedArea > int((mapArea*150)/100)):
		calcwidth -= 1
		calcheight -= 1
		calculatedArea = calcwidth * calcheight

	#and make sure the map isn't too small		
	calculatedArea = calcwidth * calcheight
	while (calculatedArea < int((mapArea*90)/100)):
		calcwidth += 1
		calcheight += 1
		calculatedArea = calcwidth * calcheight
	
	#finally, recheck to make sure our width and height are legal
	if calcwidth > 50:
		calcwidth = 50
	if calcwidth < 6:
		calcwidth = 6
	if calcheight > 60:
		calcheight = 60
	if calcheight < 6:
		calcheight = 6
		
	#allow for overrides of the width and height
	overrideWidth = getSelectedMapValue("Override Width:")
	overrideHeight = getSelectedMapValue("Override Height:")
	
	if overrideWidth != -1:
		calcwidth = overrideWidth
		print "    override width",overrideWidth
	if overrideHeight != -1:
		calcheight = overrideHeight
		print "    override height",overrideHeight
	
	
	#return the computed width and height
	print "    final width",calcwidth,"height",calcheight
	OutputMessage("Python: SmartMap: Step 1 Setting Map Size: complete")
	return (calcwidth,calcheight)

##########################################################################
#Step 1a: answer back function for deciding whether or not to wrap on x
#You must override neither or both, and SmartMap must do so because it uses
#a user-selected option to control map wrapping
def getWrapX():
	lwrapX, lwrapY = getSelectedMapValue("Wrap:")
	print "    getWrapX: wrapX", lwrapX, "wrapY", lwrapY
	return lwrapX 

##########################################################################
#Step 1b: answer back function for deciding whether or not to wrap on y
#You must override neither or both, and SmartMap must do so because it uses
#a user-selected option to control map wrapping
def getWrapY():
	lwrapX, lwrapY = getSelectedMapValue("Wrap:")
	print "    getWrapY: wrapX", lwrapX, "wrapY", lwrapY
	return lwrapY 
	
##########################################################################
#Step 2: lay out the plot types, which is a sort of broad view of the map tiles
#basically telling you if you are looking at ocean, land, hill, or mountain
def generatePlotTypes():
	beforeStepTwo()
	global wrapX
	global wrapY
	global width
	global height

	OutputMessage("Python: SmartMap: Step 2 Setting Plot Types")
	print "    in step two width",width,"height",height,"wrapxy",wrapX, wrapY
	gc = CyGlobalContext()
	cymap = CyMap()
	dice = gc.getGame().getMapRand()
	widthLimit = width-1 #range of array is 0..width-1
	heightLimit = height-1 #range of array is 0..height-1
	plotTypes = [PlotTypes.PLOT_OCEAN] * (width*height)
	ownerContinents = [-1] * (width*height)
	canBeLandLists = []
	
	#total number of useful plots on the map, this factors in staying away from the edges
	usefulPlotCount = int((width-5)*(height-5))
	
	#helpful information
	halfWidth = int(width/2)
	halfHeight = int(height/2)
	
	#get the user entered climate info
	climate = cymap.getClimate()
	climateArid = gc.getInfoTypeForString("CLIMATE_ARID")
	climateRocky = gc.getInfoTypeForString("CLIMATE_ROCKY")
	
	# Sea Level adjustment (from user input), transformed into width of sea separating continents
	sea = gc.getSeaLevelInfo(cymap.getSeaLevel()).getSeaLevelChange()
	seaWidth = 4 #default width for ocean separation of continents
	if (sea > 0):
		seaWidth = 7 #high sea level means extra separation
	if (sea < 0):
		seaWidth = 0 #low sea level means continents are permitted to merge
	print "    seawidth",seaWidth
	
	#generate continents
	continentLists = []
	canBeLandLists = []
	continentCount = getSelectedMapValue("Continents:")
	
	players = gc.getGame().countCivPlayersEverAlive()
	
	if continentCount == -1:
		continentCount = 1 + dice.get(6, "Python: SmartMap: Dice: Random Continent Count 1-6") 
	elif continentCount == -2:
		continentCount = 2 + dice.get(4, "Python: SmartMap: Dice: Random Continent Count 2-5") 
	elif continentCount == -3:
		continentCount = 3 + dice.get(6, "Python: SmartMap: Dice: Random Continent Count 3-8") 
	elif continentCount == -4:
		continentCount = 4 + dice.get(8, "Python: SmartMap: Dice: Random Continent Count 4-12") 
	elif continentCount == -5:
		continentCount = 2 + dice.get(6, "Python: SmartMap: Dice: Random Continent Count 2-7") 
	elif continentCount == -6:
		continentCount = 1 + dice.get(12, "Python: SmartMap: Dice: Random Continent Count 1-12") 
	elif continentCount == -7:
		continentCount = players
	
	print "    continentCount",continentCount
	
	#make a list for each continent
	continent = 0
	while (continent < continentCount):
		#print "    add Continent List",continent
		continent += 1
		continentLists.append([])
		canBeLandLists.append([])
		
	tries = 0
	#create a starting plot for each continent
	for contIndex in range(len(continentLists)):
		continentList = continentLists[contIndex]
		conflict = True
		while (tries < 5000 and conflict):
			tries += 1
			initX = int(width/8) + dice.get((width-int(width/8)), "Python: SmartMap: Dice: Random Continent X")
			initY = int(height/8) + dice.get((height-int(height/8)), "Python: SmartMap: Dice: Random Continent Y")
			#always grow a pangea from the center
			if continentCount == 1 and tries < 2:
				initX = int(width/2)
				initY = int(height/2)
			
			conflict = False
			for otherList in continentLists:
				if continentList != otherList:
					for otherPlot in otherList:
						diffX,diffY = getWrapDistanceXY((initX,initY),otherPlot)
						#continents must start with a certain minimum spacing
						if diffX <= seaWidth+5 and diffY <= seaWidth+5:
							conflict = True
							break
			
			if not conflict:				
				indexA = getTileIndexAtXY(initX,initY)
				if plotTypes[indexA] == PlotTypes.PLOT_OCEAN:
					continentList.append((initX,initY))
					plotTypes[indexA] = PlotTypes.PLOT_LAND
					ownerContinents[indexA] = contIndex
					print "    initial tile addition",continentList,indexA
					for nearDir in nearDirs:
						usablePlotX,usablePlotY = getTileCoordXYWithWrap((initX,initY),nearDir)
						usableIndex = getTileIndexAtXY(usablePlotX,usablePlotY)
						if plotTypes[usableIndex] == PlotTypes.PLOT_OCEAN:
							canBeLandLists[contIndex].append((usablePlotX,usablePlotY))
							
					print "    canbeland",canBeLandLists[contIndex]
					break
						
	#remove any continent that couldn't be placed
	done = False
	while not done:
		done = True
		indexLength = len(continentLists)
		for indexB in range(0,indexLength):
			continentList = continentLists[indexB]
			if len(continentList) <= 0:
				done = False
				del continentLists[indexB]
				del canBeLandLists[indexB]
				break
				
	print "    placed continents", len(continentLists)
	#find out the land style
	(roundFactor,fragPPT) = getSelectedMapValue("Land Style:")
	print "    roundFactor", roundFactor
	print "    fragPPT", fragPPT
	
	
	#generate land plots
	percentOcean = getSelectedMapValue("Ocean:")
	if percentOcean == -1:
		percentOcean = 40 + dice.get(41, "Python: SmartMap: Dice: Random Ocean %")
	if percentOcean == -2:
		percentOcean = 80 + dice.get(11, "Python: SmartMap: Dice: Random Ocean %")
	if percentOcean == -3:
		percentOcean = 60 + dice.get(26, "Python: SmartMap: Dice: Random Ocean %")
		
	if climate == climateArid:
		percentOcean -= 10
	if climate == climateRocky:
		percentOcean -= 5
		
	print "    percentOcean",percentOcean
	landPlotCount = int(((100-percentOcean) * usefulPlotCount) / 100)
	print "    landPlotCount",landPlotCount,"from total tiles:", usefulPlotCount
	
	fragLandCount = int((landPlotCount*fragPPT)/1000) 
	landPlotCount -= fragLandCount
	tries = 0
	#tries is used to ensure that this can't loop forever
	landPlaced = 0
	seaWidthCheckCap = seaWidth + 3
	while (landPlaced < landPlotCount and tries < 7 * landPlotCount):
		for contIndex in range(len(continentLists)):
			#print "cont",contIndex,"canbeland",canBeLandLists[contIndex]
			tries+=1
			if canBeLandLists[contIndex] == []:
				continue
			indexListA = dice.get(len(canBeLandLists[contIndex]), "Python: SmartMap: Dice: Random Continent listIndex backup")
			newX, newY = canBeLandLists[contIndex][indexListA]
			index = getTileIndexAtXY(newX,newY)
			if plotTypes[index] != PlotTypes.PLOT_OCEAN:
				del canBeLandLists[contIndex][indexListA]
				continue
				
			#print "cont",contIndex,"canbeland",canBeLandLists[contIndex],"considering",newX,newY
			
			dropProb = 0
			anyMatch = False
			
			#low probability of allowing approach to x-pole if not wrapping
			if not anyMatch: #I know this is always going to pass, but I do it for code symmetry
				if not wrapX:
					xFromEdge = newX - 2
					if newX > halfWidth:
						xFromEdge = (widthLimit - newX) - 2
					if xFromEdge < 20:
						edgeVal = dice.get(10000, "Python: SmartMap: Dice: Random Continent x-edge bending")
						if edgeVal < (25 * (20-xFromEdge) * (20-xFromEdge)):
							dropProb = (10 * (20-xFromEdge) * (20-xFromEdge))
							#print "dropProb",dropProb
							anyMatch = True
							#print "    drop due to x edge",newX,newY,edgeVal,xFromEdge,width

			#low probability of allowing approach to y-pole if not wrapping
			if not anyMatch:
				if not wrapY:
					yFromEdge = newY - 2
					if newY > halfHeight:
						yFromEdge = (heightLimit - newY) - 2
					if yFromEdge < 20:
						edgeVal = dice.get(10000, "Python: SmartMap: Dice: Random Continent y-edge bending")
						edgeProb = (25 * (20-yFromEdge) * (20-yFromEdge))
						#print "yFromEdge",yFromEdge,"edgeProb",edgeProb,"edgeVal",edgeVal
						if edgeVal < edgeProb:
							dropProb = (10 * (20-yFromEdge) * (20-yFromEdge))
							#print "dropProb",dropProb
							anyMatch = True
							#print "    drop due to y edge",newX,newY,edgeVal,yFromEdge,height
	
			#make sure this ocean plot doesn't touch a different continent's land
			if not anyMatch:
				newXY = (newX,newY)
				#alternative (maybe faster) method: see if any plot in -seawidth to +seawidth in x and y
				#is land and also belongs to another continent, then give further consideration
				for addX in range(-seaWidthCheckCap,seaWidthCheckCap+1):
					for addY in range(-seaWidthCheckCap,seaWidthCheckCap+1):
						otherX,otherY = getTileCoordXYWithWrap(newXY,(addX,addY))
						#print "check plot",otherX,otherY,"from plot",newXY,"dx",addX,"dy",addY
						otherIndex = getTileIndexAtXY(otherX,otherY)
						if plotTypes[otherIndex] == PlotTypes.PLOT_OCEAN:
							continue
						if ownerContinents[otherIndex] == contIndex:
							continue
						#print "found land on other continent"
						otherPlot = (otherX,otherY)
						diffX, diffY = getWrapDistanceXY(newXY, otherPlot)
						#this is much faster than doing an iteration through the dirs
						if (((diffX <= seaWidth) and (diffY <= seaWidth))):
							anyMatch = True
							#print "    drop due to seawidth",newX,newY,edgeVal,"too close to cont",otherContIndex,otherList,otherPlot
							dropProb = 10001
							break
							
						#this is much faster than doing an iteration through the dirs
						separation = int(((diffX*diffX) + (diffY*diffY))**0.5)
						beyondSeaWidthTest = 2000 - (100 * (separation - seaWidth) * (separation - seaWidth))
						edgeVal = dice.get(10000, "Python: SmartMap: Dice: Random Continent sea edge bending")
						#print "near test edgeval",edgeVal,"edgeProb",edgeProb,"diffSum",diffSum,"nearVal",seaWidthNearVal
						if edgeVal < beyondSeaWidthTest:
							dropProb = int(beyondSeaWidthTest/2)
							anyMatch = True
							#print "    drop due to seawidth margin",newX,newY,edgeVal,beyondSeaWidthTest,"too close to cont",otherIndex,"xy",otherX,otherY
							break

					if anyMatch:
						break
						
			if anyMatch:
				drop = dice.get(10000, "Python: SmartMap: Dice: random canbeland drop")
				if drop < dropProb:
					del canBeLandLists[contIndex][indexListA]
				continue
			
			#this land plot is not going to touch any other continent so place				
			#print "    place land at x",newX,"y",newY,"width",width,"height",height,"index",index
			if index <= 0 or index >= (width*height)-2:
				print "    possibly illegal place land at x",newX,"y",newY,"width",width,"height",height,"index",index
			plotTypes[index] = PlotTypes.PLOT_LAND
			ownerContinents[index] = contIndex
			continentLists[contIndex].append((newX,newY))
			del canBeLandLists[contIndex][indexListA]
			landPlaced += 1
			roundChance = 1001 - roundFactor
			for nearDir in nearDirs:
				usablePlotX,usablePlotY = getTileCoordXYWithWrap((newX,newY),nearDir)
				usableIndex = getTileIndexAtXY(usablePlotX,usablePlotY)
				if plotTypes[usableIndex] == PlotTypes.PLOT_OCEAN and not ((usablePlotX,usablePlotY) in canBeLandLists[contIndex]):
					if ((usablePlotX,usablePlotY) in canBeLandLists[contIndex]):
						roundChance -= roundFactor
					roundTest = dice.get(1000, "Python: SmartMap: Dice: Random Continent growth bending")
					if roundTest < roundChance: 
						canBeLandLists[contIndex].append((usablePlotX,usablePlotY))
			#print "    land tile addition",continentList
		
	#fill in any one tile lakes caused by wierd growth patterns
	print "    remove single tile lakes"
	restoreLake = 0
	for x in range(width):
		for y in range(height):
			countNearLand = 0
			tileIndex = getTileIndexAtXY(x,y)
			if plotTypes[tileIndex] != PlotTypes.PLOT_OCEAN:
				continue
			for nearDir in cardinalDirs:
				ox,oy = getTileCoordXYWithWrap((x,y),nearDir)
				oIndex = getTileIndexAtXY(ox,oy)
				oType = plotTypes[oIndex]
				if ((ox != x) or (oy != y)) and oType == PlotTypes.PLOT_LAND:
					countNearLand += 1
			if countNearLand == 4:
				plotTypes[tileIndex] = PlotTypes.PLOT_LAND
				restoreLake += 1
				continue
			countNearLand = 0
			for nearDir in nearDirs:
				ox,oy = getTileCoordXYWithWrap((x,y),nearDir)
				oIndex = getTileIndexAtXY(ox,oy)
				oType = plotTypes[oIndex]
				if ((ox != x) or (oy != y)) and oType == PlotTypes.PLOT_LAND:
					countNearLand += 1
			if countNearLand >=6:
				plotTypes[tileIndex] = PlotTypes.PLOT_LAND
				restoreLake += 1
				continue
				
	print "    placed",landPlaced,"land"
	global totalLandCount
	totalLandCount = landPlaced
	
	#place some random land fragments
	print "    try to place",fragLandCount,"land fragment tiles"
	#tries is used to ensure that this can't loop forever
	tries=0
	fragPlaced = 0
	usedLakeLocs = []
	fragCount = 0
	while (fragPlaced < fragLandCount and tries < 8 * fragLandCount):
		tries+=1
		fragX = dice.get(width, "Python: SmartMap: Dice: Random land Fragment X")
		fragY = dice.get(height, "Python: SmartMap: Dice: Random land Fragment Y")
		if tooCloseToPole(fragX,fragY,5):
			continue
		fragIndex = getTileIndexAtXY(fragX,fragY)
		if plotTypes[fragIndex] == PlotTypes.PLOT_LAND:
			continue
		
		allWater = True
		for nearDir in nearDirs:
			testX, testY = getTileCoordXYWithWrap((fragX,fragY),nearDir)
			if not tooCloseToPole(testX,testY,4):
				testIndex = getTileIndexAtXY(testX,testY)
				if plotTypes[testIndex] != PlotTypes.PLOT_OCEAN:
					allWater = False
					break
		if not allWater:
			continue
				
		listToAddIndex = dice.get(len(continentLists), "Python: SmartMap: Dice: Random Continent frag growth cont assignment")	
		continentLists[listToAddIndex].append((fragX,fragY))
		plotTypes[fragIndex] = PlotTypes.PLOT_LAND
		fragCount += 1
		fragPlaced += 1
		fragSize = 1
		growFrag = dice.get(100, "Python: SmartMap: Dice: Random Continent frag growth")
		gx,gy = fragX,fragY
		passes = 0
		while growFrag < 99-int(passes/5) and fragPlaced < fragLandCount:
			passes+=1
			growFrag = dice.get(100, "Python: SmartMap: Dice: Random Continent frag growth")
			dirIndex = dice.get(len(nearDirs), "Python: SmartMap: Dice: Random Continent lake growth")
			gx, gy = getTileCoordXYWithWrap((gx,gy), nearDirs[dirIndex])
			gIndex = getTileIndexAtXY(gx,gy)
			if plotTypes[gIndex] == PlotTypes.PLOT_OCEAN:
				plotTypes[gIndex] = PlotTypes.PLOT_LAND
				fragPlaced += 1
				fragSize += 1
				continentLists[listToAddIndex].append((gx,gy))
				#print "    frag",fragCount,"grew to size",fragSize
		
							
	print "    placed",fragPlaced,"frag tiles","out of required",fragLandCount,"in",fragCount,"chunks"
	
	#for continentList in continentLists:
	#	print continentList
	
	#generate hills on land plots
	percentHills = getSelectedMapValue("Hills:")
	if percentHills == -1:
		percentHills = 4 + dice.get(4, "Python: SmartMap: Dice: few Random Hills %") 
	if percentHills == -2:
		percentHills = 14 + dice.get(8, "Python: SmartMap: Dice: few Random Hills %") 
	if percentHills == -3:
		percentHills = 8 + dice.get(6, "Python: SmartMap: Dice: few Random Hills %") 
	if climate == climateRocky:
		percentHills += 3
		
	print "    percentHills",percentHills
	hillPlotCount = int((percentHills) * landPlaced / 100)
	print "    hillPlotCount",hillPlotCount,"from total tiles:", landPlaced
	firstPlotCount = int(2*hillPlotCount/3)
	
	#first portion of hills are placed randomly
	tries=0
	#tries is used to ensure that this can't loop forever
	hillPlaced = 0
	while (hillPlaced < firstPlotCount and tries < 4 * firstPlotCount):
		tries+=1
		for continentList in continentLists:
			indexListB = dice.get(len(continentList), "Python: SmartMap: Dice: Random Continent listIndex")
			existingX, existingY = continentList[indexListB]
			if not tooCloseToPole(existingX, existingY, 2):
				index = getTileIndexAtXY(existingX,existingY)
				if plotTypes[index] == PlotTypes.PLOT_LAND:
					plotTypes[index] = PlotTypes.PLOT_HILLS
					hillPlaced += 1
					#print "    land tile addition",continentList
		

	#second portion of hills must touch a hill
	tries=0
	#tries is used to ensure that this can't loop forever
	secondPlotCount = int(1*hillPlotCount/3)
	secondHillPlaced = 0
	while (secondHillPlaced < secondPlotCount and tries < 5 * secondPlotCount):
		tries+=1
		for continentList in continentLists:
			indexListC = dice.get(len(continentList), "Python: SmartMap: Dice: Random Continent listIndex")
			existingX, existingY = continentList[indexListC]
			if not tooCloseToPole(existingX, existingY, 2):
				existingIndex = getTileIndexAtXY(existingX, existingY)
				if plotTypes[existingIndex] != PlotTypes.PLOT_LAND:
					continue
				for nearDir in nearDirs:
					testX,testY = getTileCoordXYWithWrap(continentList[indexListC],nearDir)
					if not tooCloseToPole(testX, testY, 2):
						testIndex = getTileIndexAtXY(testX,testY)
						if plotTypes[testIndex] == PlotTypes.PLOT_HILLS:
							plotTypes[existingIndex] = PlotTypes.PLOT_HILLS
							secondHillPlaced += 1
							break #don't have to look any further


	totalHillsPlaced = hillPlaced + secondHillPlaced
	print "    placed",totalHillsPlaced,"hills"
	#generate hills on land plots
	percentPeaks = getSelectedMapValue("Peaks:")
	if percentPeaks == -1:
		percentPeaks = 1 + dice.get(5, "Python: SmartMap: Dice: Random Peaks %") 
	if percentPeaks == -2:
		percentPeaks = 5 + dice.get(9, "Python: SmartMap: Dice: Random Peaks %") 
	if percentPeaks == -3:
		percentPeaks = 3 + dice.get(7, "Python: SmartMap: Dice: Random Peaks %") 
	if climate == climateRocky:
		percentPeaks += 2
	print "    percentPeaks",percentPeaks
	peakPlotCount = int((percentPeaks) * landPlotCount / 100)
	print "    peakPlotCount",peakPlotCount,"from total tiles:", landPlotCount

	#peaks must touch a hill
	tries=0
	#tries is used to ensure that this can't loop forever
	peakPlaced = 0
	while (peakPlaced < peakPlotCount and tries < 5 * peakPlotCount):
		tries+=1
		for continentList in continentLists:
			indexListD = dice.get(len(continentList), "Python: SmartMap: Dice: Random Continent listIndex")
			existingX, existingY = continentList[indexListD]
			existingIndex = getTileIndexAtXY(existingX,existingY)
			if plotTypes[existingIndex] != PlotTypes.PLOT_LAND and plotTypes[existingIndex] != PlotTypes.PLOT_HILLS:
				continue
			for nearDir in nearDirs:
				testX, testY = getTileCoordXYWithWrap((existingX,existingY),nearDir)
				if not tooCloseToPole(testX,testY,2):
					testIndex = getTileIndexAtXY(testX,testY)
					if plotTypes[testIndex] == PlotTypes.PLOT_HILLS: 
						plotTypes[existingIndex] = PlotTypes.PLOT_PEAK
						peakPlaced += 1
						break #don't have to consider any more dirs
	#				print "    land tile addition",continentList
	print "    placed",peakPlaced,"peaks"
	
	global sizeSelected
	#place some random lakes
	lakePlotCount = int((8+sizeSelected) * landPlotCount / 1000) + int(restoreLake/5)
	print "    try to place",lakePlotCount,"lake tiles"
	#tries is used to ensure that this can't loop forever
	tries=0
	lakePlaced = 0
	usedLakeLocs = []
	while (lakePlaced < lakePlotCount and tries < 8 * lakePlotCount):
		tries+=1
		for continentList in continentLists:
			indexListE = dice.get(len(continentList), "Python: SmartMap: Dice: Random Continent listIndex")
			existingX, existingY = continentList[indexListE]
			existingIndex = getTileIndexAtXY(existingX,existingY)
			if tooCloseToPole(existingX,existingY,4):
				continue
			if plotTypes[existingIndex] == PlotTypes.PLOT_LAND:
				#is there is a lake too close already
				tooClose = False
				for usedLakeLoc in usedLakeLocs:
					distX,distY = getWrapDistanceXY((existingX, existingY),usedLakeLoc)
					if distX < 3 and distY < 3:
						tooClose = True
						break
				if tooClose:
					continue
				
				nearHillPeak = False
				allLand = True
				for nearDir in nearDirs:
					testX, testY = getTileCoordXYWithWrap((existingX,existingY),nearDir)
					if not tooCloseToPole(testX,testY,2):
						testIndex = getTileIndexAtXY(testX,testY)
						if plotTypes[testIndex] == PlotTypes.PLOT_OCEAN:
							allLand = False
							break
						if plotTypes[testIndex] == PlotTypes.PLOT_HILLS or plotTypes[testIndex] == PlotTypes.PLOT_PEAK:
							nearHillPeak = True
							break
				if allLand and nearHillPeak:
					plotTypes[existingIndex] = PlotTypes.PLOT_OCEAN
					lakePlaced += 1
					growLake = dice.get(100, "Python: SmartMap: Dice: Random Continent lake growth")
					lakeList = [(existingX,existingY)]
					lakeSize = 1
					usedLakeLocs.append((existingX,existingY))
					gx,gy = existingX,existingY
					passes = 0
					while growLake < (80 + (4*sizeSelected) - int(passes/5)) and lakePlaced < lakePlotCount:
						passes += 1
						growLake = dice.get(100, "Python: SmartMap: Dice: Random Continent lake growth")
						dirIndex = dice.get(len(cardinalDirs), "Python: SmartMap: Dice: Random Continent lake growth")
						gx, gy = getTileCoordXYWithWrap((gx,gy), cardinalDirs[dirIndex])
						gIndex = getTileIndexAtXY(gx,gy)
						if plotTypes[gIndex] != PlotTypes.PLOT_LAND:
							continue
						nearLandCount = 0
						for nearDir in nearDirs:
							testX, testY = getTileCoordXYWithWrap((gx,gy),nearDir)
							if not tooCloseToPole(testX,testY,2):
								testIndex = getTileIndexAtXY(testX,testY)
								if plotTypes[testIndex] != PlotTypes.PLOT_OCEAN:
									nearLandCount += 1
								elif (testX,testY) in lakeList:
									nearLandCount += 1
						#only grow if we are touching all land or existing parts of lake
						if nearLandCount == 8:
							plotTypes[gIndex] = PlotTypes.PLOT_OCEAN
							lakeList.append((gx,gy))
							usedLakeLocs.append((gx,gy))
							lakeSize += 1
							lakePlaced += 1
							#print "    lake",lakePlaced,"grew to size", lakeSize
							
	print "    placed",lakePlaced,"lake tiles","out of required",lakePlotCount
	
	#force ocean near poles
	for x in range(width):
		for y in range(height):
			if tooCloseToPole(x,y,2):
				indexOfPlot = getTileIndexAtXY(x,y)
				plotTypes[indexOfPlot] = PlotTypes.PLOT_OCEAN
	
	#insert a transformation to put the widest vertical ocean on the x-wrap
	transPlotTypes = plotTypes[:]			
	if wrapX:
		maxVerticalOcean = 0
		verticalOceanLocation = [1]
		for x in range(width):
			oceanForX = 0
			for xtally in (-2,-1,0,1,2):
				for y in range(height):
					checkX,checkY = getTileCoordXYWithWrap((x,y),(xtally,0))
					plotIndex = getTileIndexAtXY(checkX,checkY)
					if plotTypes[plotIndex] == PlotTypes.PLOT_OCEAN:
						oceanForX += 1
			if oceanForX > maxVerticalOcean:
				maxVerticalOcean = oceanForX
				verticalOceanLocation = [x]
			elif oceanForX == maxVerticalOcean:
				verticalOceanLocation.append(x)
				
		firstHalfLocs = []
		secondHalfLocs = []
		for loc in verticalOceanLocation:
			if loc < int(width/2):
				firstHalfLocs.append(loc)
			else:
				secondHalfLocs.append(loc)
				
		usedLocs = firstHalfLocs
		if len(secondHalfLocs) > len(usedLocs):
			usedLocs = secondHalfLocs
				
		usedLocs.sort()
		actualWrapLocation = usedLocs[int(len(usedLocs)/2)]
		actualWrapLocation += 1	
		print "    wrap X at",actualWrapLocation,"chosen from",usedLocs	
		#transform the resultplottypes by x
		for x in range(width):
			for y in range(height):
				readX, readY = getTileCoordXYWithWrap((x,y),(actualWrapLocation,0))
				readIndex = getTileIndexAtXY(readX,readY)
				writeIndex = getTileIndexAtXY(x,y)
				transPlotTypes[writeIndex] = plotTypes[readIndex]
				
	plotTypes = transPlotTypes

	#insert a transformation to put the widest horizontal ocean on the y-wrap
	transPlotTypes = plotTypes[:]			
	if wrapY:
		maxHorizontalOcean = 0
		horizontalOceanLocation = [1]
		for y in range(height):
			oceanForY = 0
			for ytally in (-2,-1,0,1,2):
				for x in range(width):
					checkX,checkY = getTileCoordXYWithWrap((x,y),(0,ytally))
					plotIndex = getTileIndexAtXY(checkX,checkY)
					if plotTypes[plotIndex] == PlotTypes.PLOT_OCEAN:
						oceanForY += 1
			if oceanForY > maxHorizontalOcean:
				maxHorizontalOcean = oceanForY
				horizontalOceanLocation = [y]
			elif oceanForY == maxHorizontalOcean:
				horizontalOceanLocation.append(y)
				
		firstHalfLocs = []
		secondHalfLocs = []
		for loc in horizontalOceanLocation:
			if loc < int(height/2):
				firstHalfLocs.append(loc)
			else:
				secondHalfLocs.append(loc)
				
		usedLocs = firstHalfLocs
		if len(secondHalfLocs) > len(usedLocs):
			usedLocs = secondHalfLocs
				
		usedLocs.sort()
		actualWrapLocation = usedLocs[int(len(usedLocs)/2)]
		actualWrapLocation += 1	
		print "    wrap Y at",actualWrapLocation,"chosen from",usedLocs	
		#transform the resultplottypes by y
		for x in range(width):
			for y in range(height):
				readX, readY = getTileCoordXYWithWrap((x,y),(0,actualWrapLocation))
				readIndex = getTileIndexAtXY(readX,readY)
				writeIndex = getTileIndexAtXY(x,y)
				transPlotTypes[writeIndex] = plotTypes[readIndex]
				
	plotTypes = transPlotTypes
	
	
	checkForBadPlots()
	OutputMessage("Python: SmartMap: Step 2 Setting Plot Types: complete")
	# Finished generating plots!
	return plotTypes
	
##########################################################################
#Step 3: now generate the fine terrain, in particular, need to decide if 
#any land or hill tile is desert, plains, grass, tundra, or snow
def generateTerrainTypes():	
	global width
	global height	
	
	OutputMessage("Python: SmartMap: Step 3 Generate Terrain")
	gc = CyGlobalContext()
	cymap = CyMap()
	dice = gc.getGame().getMapRand()
	
	global postContinentLists
	checkForBadPlots()
	
	#post-processing cleanup, anything like this wants to happen before river placement
	#so that river flow will make sense!
	print "    flatten peaks on coast"
	for x in range(width):
		for y in range(height):
			pPlot = cymap.plot(x,y)
			# A general map post processing example: remove all peaks along the ocean
			if pPlot.isPeak() and pPlot.isCoastalLand():
				pPlot.setPlotType(PlotTypes.PLOT_HILLS, True, True)
	
	
	preComputeAltitudes()
	computeContinents()

	#find out the user selected terrain method
	terrainMethod = getSelectedMapValue("Terrain:")
	print "    terrainMethod",terrainMethod

	if terrainMethod == "standard":
		terraingen = TerrainGenerator()
		terrainTypes = terraingen.generateTerrain()
		OutputMessage("Python: SmartMap: Step 3 Generate Terrain: complete standard")
		return terrainTypes

	if terrainMethod == "great plains":
		terraingen = GreatPlainsTerrainGenerator()
		terrainTypes = terraingen.generateTerrain()
		OutputMessage("Python: SmartMap: Step 3 Generate Terrain: complete great plains")
		return terrainTypes

	if terrainMethod == "oasis":
		terraingen = OasisTerrainGenerator()
		terrainTypes = terraingen.generateTerrain()
		OutputMessage("Python: SmartMap: Step 3 Generate Terrain: complete oasis")
		return terrainTypes
		
	#otherwise, smartmap
	#get the user entered climate info
	climate = cymap.getClimate()
	climateTemperate = gc.getInfoTypeForString("CLIMATE_TEMPERATE")
	climateTropical = gc.getInfoTypeForString("CLIMATE_TROPICAL")
	climateArid = gc.getInfoTypeForString("CLIMATE_ARID")
	climateCold = gc.getInfoTypeForString("CLIMATE_COLD")

	extraRiverPct = 100
	
	if climate == climateTropical:
		extraRiverPct += 11
		
	if climate == climateArid:
		extraRiverPct -= 11
	
	print "    generate rivers"
	#first pass: place rivers.  if you don't place rivers first
	#then oasis and floodplains are impossible
	#1 in 1000 hills or 1 in 200 peaks spawns a river in a random direction			
	dirN = gc.getInfoTypeForString("CARDINALDIRECTION_NORTH")
	dirE = gc.getInfoTypeForString("CARDINALDIRECTION_EAST")
	dirS = gc.getInfoTypeForString("CARDINALDIRECTION_SOUTH")
	dirW = gc.getInfoTypeForString("CARDINALDIRECTION_WEST")
	riverDirs = [dirN, dirE, dirS, dirW]
	riverDirTypes = [CardinalDirectionTypes.CARDINALDIRECTION_NORTH,
					CardinalDirectionTypes.CARDINALDIRECTION_EAST,
					CardinalDirectionTypes.CARDINALDIRECTION_SOUTH,
					CardinalDirectionTypes.CARDINALDIRECTION_WEST]
		
	#each continent gets at least one river from its highest point			
	for continentList in postContinentLists:
		if len(continentList) < 10:
			continue
		maxContAltitude = -1
		cx,cy = continentList[0]
		bestPlot = cymap.plot(cx,cy)
		for plot in continentList:
			plotX,plotY = plot
			pPlot = cymap.plot(plotX,plotY)
			altitude = getRiverAltitude([pPlot])
			if altitude > maxContAltitude:
				maxContAltitude = altitude
				bestPlot = pPlot
		riverDirIndex = dice.get(len(riverDirs), "Python: SmartMap: Dice: High River Placement")
		CyMapGenerator().doRiver(bestPlot, riverDirTypes[riverDirIndex])
			
	#additional random rivers			
	for rx in range(width):
		for ry in range(height):
			pPlot = cymap.plot(rx,ry)
			if pPlot.isRiver() or pPlot.isCoastalLand():
				continue
			otherRiver = False
			for rNearDir in playerStartCross:
				rnewX, rnewY = getTileCoordXYWithWrap((rx,ry), rNearDir)
				otherPlot = cymap.plot(rnewX,rnewY)
				if otherPlot.isRiver():
					otherRiver = True
					break
					
			if otherRiver:
				continue
				
			plotAltitude = int(getRiverAltitude([pPlot]) / 2.5) #range is 0 to 100
			percentFromEquatorXY = percentFromEquator(rx,ry)
			percentFromPoleXY = percentFromPole(rx,ry)
			riverOdds = dice.get(10000, "Python: SmartMap: Dice: River Placement Odds")
			if ( (pPlot.isHills() and riverOdds < int(((150+(2*percentFromEquatorXY)+(2*percentFromPoleXY)+(2*plotAltitude))*extraRiverPct)/100)) or 
			     (pPlot.isPeak() and riverOdds < int(((1600+(10*percentFromEquatorXY)+(10*percentFromPoleXY)+(30*plotAltitude))*extraRiverPct)/100))
			     ):
				riverDirIndex = dice.get(len(riverDirs), "Python: SmartMap: Dice: River Placement")
				CyMapGenerator().doRiver(pPlot, riverDirTypes[riverDirIndex])
	
	
	extraDesert = 0
	extraCold = 0
	
	if climate == climateArid:
		extraDesert += 30
		extraCold -= 6
	if climate == climateCold:
		extraDesert -= 16
		extraCold += 16
	if climate == climateTemperate:
		extraDesert -= 18
		extraCold -= 6
	if climate == climateTropical:
		extraDesert -= 24
		extraCold -= 12

	coverageSetting = getSelectedMapValue("Forest/Jungle:")
	if coverageSetting == "heavy":
		extraDesert -= 10

	
	print "    generate terrain"
	#default generation method: SmartMap terrain
	terrainData = [TerrainTypes.NO_TERRAIN]*(width*height)
	
	terrainDesert = gc.getInfoTypeForString("TERRAIN_DESERT")
	terrainPlains = gc.getInfoTypeForString("TERRAIN_PLAINS")
	terrainGrass = gc.getInfoTypeForString("TERRAIN_GRASS")
	terrainTundra = gc.getInfoTypeForString("TERRAIN_TUNDRA")
	terrainSnow = gc.getInfoTypeForString("TERRAIN_SNOW")
	
	global preComputedWetness
	preComputeWetness()
	
	#for every terrain we need to consider in x,y
	for x in range(width):
		for y in range(height):
			pPlot = cymap.plot(x,y)
			index = getTileIndexAtXY(x,y)
			#make sure every plot has a default set
			terrainData[index] = pPlot.getTerrainType()
			#water tiles, peaks, and hills should always take their default terrain type
			if pPlot.isWater():
				terrainData[index] = pPlot.getTerrainType()
				continue
			if pPlot.isPeak():	
				terrainData[index] = pPlot.getTerrainType()
				continue
			
			#land and hill tiles actually have choices in what type of tile to be
			plotType = pPlot.getPlotType()
			if plotType == PlotTypes.PLOT_LAND or plotType == PlotTypes.PLOT_HILLS:
				#note that height is at the top of the map (northmost), when you might expect bottom
				#this allows 10% snow only near the very furthest north or south strip of land
				percentFromPoleXY = percentFromPole(x,y)
				percentFromEquatorXY = percentFromEquator(x,y)
				#print "    eq%",percentFromEquatorXY,"po%",percentFromPoleXY,"x",x,"y",y
				#when very close to the pole, you get either snow, tundra, or plains
				plotAltitudeFactor = -8 + int(getRiverAltitude([pPlot]) / 30) #range is -8 to 8

				#water content of this tile
				waterFactor = int(preComputedWetness[index] / 2.5)
				isWater = False
				if pPlot.isFreshWater() or pPlot.isRiver():
					isWater = True

				snowRand = dice.get(100, "Python: SmartMap: Dice: Random Terrain")
				tundraRand = dice.get(100, "Python: SmartMap: Dice: Random Terrain")
				if percentFromPoleXY<23+extraCold and snowRand < 5*(23+extraCold-percentFromPoleXY+plotAltitudeFactor):
					terrainData[index] = terrainSnow
					continue
				if percentFromPoleXY<23+extraCold and tundraRand < 68+plotAltitudeFactor:
					terrainData[index] = terrainTundra
					continue
				if percentFromPoleXY<23+extraCold:
					terrainData[index] = terrainPlains
					continue
					
				#near the poles, but further out, tundra or plains
				if percentFromPoleXY<41+extraCold and tundraRand < 5*(41+extraCold-percentFromPoleXY+plotAltitudeFactor):
					terrainData[index] = terrainTundra
					continue
				if percentFromPoleXY<41+extraCold:
					terrainData[index] = terrainPlains
					continue

				#in central portion of map, get desert
				desertRand = dice.get(100, "Python: SmartMap: Dice: Random Terrain")
				plainsRand = dice.get(100, "Python: SmartMap: Dice: Random Terrain")
				desertSign = 1
				if extraDesert < 0:
					desertSign = -1
				desertWidthFactor = dice.get(abs(extraDesert)+1, "Python: SmartMap: Dice: Random Terrain")
				desertWidthFactor *= desertSign
				if (percentFromEquatorXY<(38+desertWidthFactor) and 
				    desertRand < 60+plotAltitudeFactor-waterFactor+extraDesert-percentFromEquatorXY+desertWidthFactor):
					terrainData[index] = terrainDesert
					continue
				if percentFromEquatorXY<(38+desertWidthFactor) and (plainsRand<50 or not isWater):
					terrainData[index] = terrainPlains
					continue
				if percentFromEquatorXY<(38+desertWidthFactor):
					terrainData[index] = terrainGrass
					continue
					
				#grass, particularly near water
				grassRand = dice.get(100, "Python: SmartMap: Dice: Random Terrain")
				if grassRand < (40 + waterFactor):
					#print "    grass eq%",percentFromEquatorXY,"po%",percentFromPoleXY,"x",x,"y",y
					terrainData[index] = terrainGrass
					continue
				#everything else plains
				terrainData[index] = terrainPlains
				continue
		
	#second pass: clumpify terrain by giving each plot a chance to match its neighbors
	allLandPlots = []
	for x in range(width):
		for y in range(height):
			pPlot = cymap.plot(x,y)
			plotType = pPlot.getPlotType()
			if plotType == PlotTypes.PLOT_LAND:
				allLandPlots.append((x,y))
				
	while (len(allLandPlots) > 0):
		pickIndex = dice.get(len(allLandPlots), "Python: SmartMap: Dice: Random Terrain Plot Pick")
		(x,y) = allLandPlots[pickIndex]
		del allLandPlots[pickIndex]
		#print "flip plot",x,y
		desertN=0
		snowN=0
		tundraN=0
		grassN=0
		plainsN=0
		for nearDir in nearDirs:
			otherX, otherY = getTileCoordXYWithWrap((x,y), nearDir)
			otherPlot = cymap.plot(otherX,otherY)
			otherPlotType = otherPlot.getPlotType()
			if otherPlotType == PlotTypes.PLOT_LAND:
				otherIndex = getTileIndexAtXY(otherX,otherY)
				otherTerrain = terrainData[otherIndex]
				if otherTerrain == terrainDesert:
					desertN += 1
				elif otherTerrain == terrainPlains:
					plainsN += 1
				elif otherTerrain == terrainGrass:
					grassN += 1
				elif otherTerrain == terrainTundra:
					tundraN += 1
				elif otherTerrain == terrainSnow:
					snowN += 1
			
		scores = [desertN,snowN,tundraN,grassN,plainsN]
		maxScore = max(scores)
		if maxScore < 3:
			continue
		skipVal = dice.get(1000, "Python: SmartMap: Dice: Random Terrain Plot Pick")
		if maxScore < 6 and skipVal > 120:
			continue #only flip a certain % of plots
		if grassN == maxScore:
			terrainData[index] = terrainGrass
		elif plainsN == maxScore:
			terrainData[index] = terrainPlains
		elif desertN == maxScore:
			terrainData[index] = terrainDesert
		elif tundraN == maxScore:
			terrainData[index] = terrainTundra
		elif snowN == maxScore:
			terrainData[index] = terrainSnow
	
	#third pass: no desert within one step of fresh water on tropical
	if climate == climateTropical:
		for x in range(width):
			for y in range(height):
				pPlot = cymap.plot(x,y)
				index = getTileIndexAtXY(x,y)
				plotType = pPlot.getPlotType()
				pickedTerrainType = terrainData[index]
				if (plotType == PlotTypes.PLOT_LAND or plotType == PlotTypes.PLOT_HILLS) and pickedTerrainType == terrainDesert:
					for nearDir in nearDirs:
						otherX, otherY = getTileCoordXYWithWrap((x,y), nearDir)
						otherPlot = cymap.plot(otherX,otherY)
						if otherPlot.isFreshWater():
							terrainData[index] = terrainPlains
							break
		
				
	#fourth pass: set all hills to match most common touching terrain
	for x in range(width):
		for y in range(height):
			pPlot = cymap.plot(x,y)
			index = getTileIndexAtXY(x,y)
			plotType = pPlot.getPlotType()
			if (plotType == PlotTypes.PLOT_HILLS):
				desertN=0
				snowN=0
				tundraN=0
				grassN=0
				plainsN=0
				for nearDir in nearDirs:
					otherX, otherY = getTileCoordXYWithWrap((x,y), nearDir)
					otherPlot = cymap.plot(otherX,otherY)
					otherPlotType = otherPlot.getPlotType()
					if otherPlotType == PlotTypes.PLOT_LAND:
						otherIndex = getTileIndexAtXY(otherX,otherY)
						otherTerrain = terrainData[otherIndex]
						if otherTerrain == terrainDesert:
							desertN += 1
						elif otherTerrain == terrainPlains:
							plainsN += 1
						elif otherTerrain == terrainGrass:
							grassN += 1
						elif otherTerrain == terrainTundra:
							tundraN += 1
						elif otherTerrain == terrainSnow:
							snowN += 1
					
				scores = [desertN,snowN,tundraN,grassN,plainsN]
				if grassN == max(scores):
					terrainData[index] = terrainGrass
				elif plainsN == max(scores):
					terrainData[index] = terrainPlains
				elif desertN == max(scores):
					terrainData[index] = terrainDesert
				elif tundraN == max(scores):
					terrainData[index] = terrainTundra
				elif snowN == max(scores):
					terrainData[index] = terrainSnow
				else:
					terrainData[index] = terrainPlains
	
	OutputMessage("Python: SmartMap: Step 3 Generate Terrain: complete")
	checkForBadPlots()
	return terrainData
	
##########################################################################
#Step 3A: add rivers.  since smartmap handles this elsewhere we must override
def addRivers():
	OutputMessage("Python: SmartMap: Step 3A rivers")
	checkForBadPlots()

	#find out the user selected terrain method
	terrainMethod = getSelectedMapValue("Terrain:")
	print "    riverMethod",terrainMethod

	standardRiverMethods = ["standard","great plains","oasis"]
	if terrainMethod in standardRiverMethods:
		CyPythonMgr().allowDefaultImpl()
		OutputMessage("Python: SmartMap: Step 3A rivers: complete standard")
		return
	checkForBadPlots()
	OutputMessage("Python: SmartMap: Step 3A rivers: complete")
	return
	
##########################################################################
#Step 3B: add lakes.  since smartmap handles this elsewhere we must override
def addLakes():
	OutputMessage("Python: SmartMap: Step 3B lakes")
	checkForBadPlots()

	#find out the user selected terrain method
	terrainMethod = getSelectedMapValue("Terrain:")
	print "    lakeMethod",terrainMethod

	standardLakeMethods = ["standard","great plains","oasis"]
	if terrainMethod in standardLakeMethods:
		CyPythonMgr().allowDefaultImpl()
		OutputMessage("Python: SmartMap: Step 3B lakes: complete standard")
		return
	checkForBadPlots()
	OutputMessage("Python: SmartMap: Step 3B lakes: complete")
	return

##########################################################################
#Step 4: add features to your terrain	
#this is responsible for things like oasis, ice floes on the ocean, floodplains
def addFeatures():
	global wrapX
	global wrapY
	global width
	global height
	OutputMessage("Python: SmartMap: Step 4 Feature Generation")
	checkForBadPlots()
	gc = CyGlobalContext()
	cymap = CyMap()
	dice = gc.getGame().getMapRand()

	#find out the user selected terrain method
	featureMethod = getSelectedMapValue("Features:")
	print "    featureMethod",featureMethod
	global preComputedWetness
	preComputeWetness()

	global featuregen
	if featureMethod == "standard":
		featuregen = FeatureGenerator()
		featuregen.addFeatures()
		OutputMessage("Python: SmartMap: Step 4 Feature Generation: complete standard")
		return 0

	if featureMethod == "great plains":
		featuregen = GreatPlainsFeatureGenerator()
		featuregen.addFeatures()
		OutputMessage("Python: SmartMap: Step 4 Feature Generation: complete great plains")
		return 0

	if featureMethod == "oasis":
		featuregen = OasisFeatureGenerator()
		featuregen.addFeatures()
		OutputMessage("Python: SmartMap: Step 4 Feature Generation: complete oasis")
		return 0
		
	#defaults to SmartMap mode
	strict = False #determines whether to allow things like floodplains/plains
	if featureMethod == "SmartMap Strict":
		strict = True
		
	#get the user entered climate info
	climate = cymap.getClimate()
	climateTemperate = gc.getInfoTypeForString("CLIMATE_TEMPERATE")
	climateTropical = gc.getInfoTypeForString("CLIMATE_TROPICAL")
	climateArid = gc.getInfoTypeForString("CLIMATE_ARID")
	climateCold = gc.getInfoTypeForString("CLIMATE_COLD")
	
	extraForest,extraJungle = getSelectedMapValue("Forest/Jungle:")
	extraCold = 0
	extraOasis = 0
	
	if climate == climateTropical:
		extraJungle += 10
		extraOasis += 3
		
	if climate == climateTemperate:
		extraForest += 10

	if climate == climateCold:
		extraForest += 10
		extraCold += 2

	if climate == climateArid:
		extraJungle -= 10
		extraOasis -= 2
		
	featIce = gc.getInfoTypeForString("FEATURE_ICE")
	featJungle = gc.getInfoTypeForString("FEATURE_JUNGLE")
	featOasis = gc.getInfoTypeForString("FEATURE_OASIS")
	featFlood = gc.getInfoTypeForString("FEATURE_FLOOD_PLAINS")
	featForest = gc.getInfoTypeForString("FEATURE_FOREST")
	terrainDesert = gc.getInfoTypeForString("TERRAIN_DESERT")
	terrainTundra = gc.getInfoTypeForString("TERRAIN_TUNDRA")


	print "    generate smartmap features"
	for x in range(width):
		for y in range(height):
			#force ice at poles
			pPlot = cymap.plot(x,y)
			if tooCloseToPole(x,y,1):
				pPlot.setFeatureType(featIce,-1)
				continue

			percentFromPoleXY = percentFromPole(x,y)
			percentFromEquatorXY = percentFromEquator(x,y)
			terrainType = pPlot.getTerrainType()
			plotIndex = getTileIndexAtXY(x,y)
			waterFactor = preComputedWetness[plotIndex]

			#ice at the poles, for whatever poles this planet has
			iceOdds = dice.get(1000, "Python: SmartMap: Dice: Feature Placement Odds")
			if pPlot.isWater() and percentFromPoleXY<=13+extraCold and pPlot.canHaveFeature(featIce) and iceOdds < 100*(13-percentFromPoleXY+extraCold):
				pPlot.setFeatureType(featIce,-1)
				continue
				
			#no other features exist on water
			if pPlot.isWater():
				continue
				
			#oasis don't ask if it canHaveFeature oasis because that always returns False
			oasisOdds = dice.get(1000, "Python: SmartMap: Dice: Feature Placement Odds")
			oasisBonus = 0
			if terrainType == terrainDesert:
				oasisBonus += 15
			if (pPlot.isFlatlands() and not pPlot.isFreshWater() and oasisOdds<5+extraOasis+oasisBonus and 
			    (terrainType == terrainDesert or terrainType == terrainTundra)):
				if not strict or pPlot.canHaveFeature(featOasis):
					pPlot.setFeatureType(featOasis,-1)
					continue

			#flood same issue as oasis, always False for some reason
			floodOdds = dice.get(1000, "Python: SmartMap: Dice: Feature Placement Odds")
			floodBonus = int(waterFactor / 10)
			if pPlot.getTerrainType() == terrainDesert:
				floodBonus = 360 + waterFactor
			if pPlot.isFreshWater() and floodOdds<(40+(5*extraJungle)+floodBonus):
				if not strict or pPlot.canHaveFeature(featFlood):
					pPlot.setFeatureType(featFlood,-1)
					continue

			#jungle
			jungleOdds = dice.get(1000, "Python: SmartMap: Dice: Feature Placement Odds")
			if (percentFromEquatorXY < 60+extraJungle and pPlot.canHaveFeature(featJungle) and 
			    jungleOdds < int((40+extraJungle+int(waterFactor/20)) * (100-percentFromEquatorXY)/10) ):
				pPlot.setFeatureType(featJungle,-1)
				continue
	
			#forest
			forestOdds = dice.get(1000, "Python: SmartMap: Dice: Feature Placement Odds")
			if (percentFromEquatorXY > 45-extraForest and pPlot.canHaveFeature(featForest) and 
			    forestOdds < int((40+extraForest+int(waterFactor/20)) * (percentFromEquatorXY)/10)):
				pPlot.setFeatureType(featForest,-1)
				continue
		
	OutputMessage("Python: SmartMap: Step 4 Feature Generation: complete")
	checkForBadPlots()
	return 0
	
##########################################################################
#Step 5: add bonuses to your map
#this method puts down the tile bonuses, such as iron, whale, rice, oil, etc.	
def addBonuses():
	global width
	global height
	
	OutputMessage("Python: SmartMap: Step 5 Add bonuses")
	checkForBadPlots()

	gc = CyGlobalContext()
	cymap = CyMap()
	dice = gc.getGame().getMapRand()

	#find out the user selected terrain method
	bonusMethod = getSelectedMapValue("Bonuses:")
	print "    bonusMethod",bonusMethod

	if bonusMethod == "standard":
		CyPythonMgr().allowDefaultImpl()
		OutputMessage("Python: SmartMap: Step 5 Add bonuses: complete standard")
		return
	#Otherwise, smartmap placement
		
	global postContinentLists

	#these are the resources you find in water tiles
	resourcesWater = ('BONUS_CLAM', 'BONUS_FISH', 'BONUS_WHALE', 'BONUS_CRAB')
	#these resources should be given more often than others
	resourcesCommon = ('BONUS_STONE', 'BONUS_MARBLE', 'BONUS_CORN', 
	'BONUS_COW', 'BONUS_DEER', 'BONUS_PIG', 'BONUS_RICE', 
	'BONUS_SHEEP', 'BONUS_WHEAT', 'BONUS_FISH', 'BONUS_WHALE')
	
	#for whatever reason, these resources never validate as canHaveBonus so we override
	#and place them on any empty hill
	resourcesEmptyHill = ('BONUS_ALUMINUM','BONUS_GOLD','BONUS_SILVER')
	global totalLandCount
	#print "    totalLandCount",totalLandCount
	players = gc.getGame().countCivPlayersEverAlive()
	
	# Set plot eligibility for current bonus.
	# Begin by initiating the list of lists, into which eligible plots will be recorded.
	eligible = []
	for bonusId in range(gc.getNumBonusInfos()):
		eligible.append([])
	
	#find all eligible plots for each bonus
	for x in range(width):
		for y in range(height):
			pPlot = cymap.plot(x,y)
			if pPlot.getBonusType(-1) != -1:
				continue # to next plot.
			if pPlot.getFeatureType() == gc.getInfoTypeForString("FEATURE_OASIS"):
				continue # Soren wants no bonuses in oasis plots. So mote it be.
			# Check plot type and features for eligibility.
			for bonusId in range(gc.getNumBonusInfos()):
				bonusInfoXML = gc.getBonusInfo(bonusId)
				type_string = bonusInfoXML.getType()
				# First check the plot for an existing bonus.
				if (pPlot.canHaveBonus(bonusId, True)):
					eligible[bonusId].append((x,y))
				if type_string in resourcesEmptyHill and pPlot.isHills():
					eligible[bonusId].append((x,y))
	
	for bonusId in range(gc.getNumBonusInfos()):
		bonusInfoXML = gc.getBonusInfo(bonusId)
		type_string = bonusInfoXML.getType()
		#print "    bonus",type_string,"per tile:",perTileRate
		perTileRate = 300
		baseRate = int(totalLandCount/perTileRate) #base rate of 1 of each resource for every tile-rate land tiles
		#find out how many players in game, and make sure there are some for each
		playerExtra = int(players*12/100) + dice.get(1+int(players * 25 / 100), "Python: SmartMap: Dice: Extra resources per player")
		extraFactor = 0
		
		contExtra = 1 + dice.get(len(postContinentLists)+1,"Python: SmartMap: Dice: Extra resources per continent")
		if contExtra > 6:
			contExtra = 6
		
		#the bountiful ocean - I add extra resources to water tiles because there
		#is a lot of coastal land in this map design
		if type_string in resourcesWater:
			extraFactor += dice.get(1+(players*140)/100, "Python: SmartMap: Dice: Extra resources in water")
			
		#some resources are meant to be a bit more common
		if type_string in resourcesCommon:
			extraFactor += int(players*12/100) + dice.get(1+int(baseRate*24/100), "Python: SmartMap: Dice: Extra resources common resources")
		
		#total is based on: size of map, num players, random extra per player, extra for water types
		#print "    resource",bonusId,"baseRate",baseRate,"playerExtra",playerExtra,"extraFactor",extraFactor
		count = int(2 + baseRate + playerExtra + extraFactor)
		#print "    type",type_string,"count",count
		
		#compute the fair shares of each resource each continent is due
		shares = []
		totalLand = 0
		for continentList in postContinentLists:
			totalLand += len(continentList)
		for continentList in postContinentLists:
			share = int((100*(len(continentList)+1))/totalLand)+1
			shares.append(share)
			
		print "    fair-share continent shares",shares
		minOfShares = min(shares)
		maxOfShares = max(shares)
		placeOneOnSmallest = int(100/minOfShares)+1
		placeTries = 0
		while placeOneOnSmallest > count and placeTries < 5:
			placeTries+=1
			nextSmallest = maxOfShares
			for share in shares:
				if share < nextSmallest and share > minOfShares:
					nextSmallest = share
			minOfShares = nextSmallest
			placeOneOnSmallest = int(100/minOfShares)+2
		
		
		added = 0
		addedLocs = []
		#this tries to give each continent a fair share of each resource
		fairShare = placeOneOnSmallest#amount to share fairly
		minFairShare = int(count/3) #we'll share half if that is more than the above
		if fairShare < minFairShare: #share this many if more than needed
			fairShare = minFairShare
		if fairShare > count: #not enough to consider sharing fairly
			fairShare = 0
		maxFairShare = int(500*len(postContinentLists)/100) #if more than this, place remainder randomly
		if fairShare > maxFairShare:
			fairShare = maxFairShare
		if (not type_string in resourcesWater):
			print "    attempting to fair-share",fairShare,"out of",count,"of",type_string,"over",len(postContinentLists),"continents"
			for contIndex in range(len(postContinentLists)):
				contShare = int( (fairShare * shares[contIndex]) / 100)
				continentList = postContinentLists[contIndex]
				#print "try to put",contShare,"of",type_string,"on",contIndex
				for shareNum in range(contShare):
					for tries in range(200):
						tileIndex = dice.get(len(continentList), "Python: SmartMap: Dice: Continent Bonus Placement")
						tx,ty = continentList[tileIndex]
						pPlot = cymap.plot(tx,ty)
						if pPlot.getBonusType(-1) != -1:
							continue
						if pPlot.getFeatureType() == gc.getInfoTypeForString("FEATURE_OASIS"):
							continue
						if (not pPlot.canHaveBonus(bonusId, True) and (not ((type_string in resourcesEmptyHill) and pPlot.isHills()))):
							continue
						tooClose = False
						for usedLoc in addedLocs:
							distX,distY = getWrapDistanceXY((tx,ty),usedLoc)
							if distX + distY < 7:
								tooClose = True
								break
						if tooClose:
							continue
							
						#print "    fair share put 1 of",type_string,"on continent",contIndex,"for shareNum",shareNum,"on try",tries
						pPlot.setBonusType(bonusId)
						added += 1
						count -= 1
						addedLocs.append((tx,ty))
						break

		# Now we assign the bonuses to eligible plots chosen completely at random
		# from among the valid plots found above
		while count > 0:
			if eligible[bonusId] == []:
				break # No eligible plots left!
			index = dice.get(len(eligible[bonusId]), "Python: SmartMap: Dice: Bonus Placement")
			px,py = eligible[bonusId][index]
			pPlot = cymap.plot(px,py)
			if pPlot.getBonusType(-1) != -1:
				del eligible[bonusId][index] # Remove this plot from the eligible list, it already had a bonus
				continue
				
			tooClose = False
			for usedLoc in addedLocs:
				distX,distY = getWrapDistanceXY((px,py),usedLoc)
				if distX + distY < 7:
					tooClose = true
					break
			if tooClose:
				del eligible[bonusId][index] # Remove this plot from the eligible list, it was too close to another bonus
				continue
				
			pPlot.setBonusType(bonusId)
			added += 1
			addedLocs.append((px,py))
			del eligible[bonusId][index] # Remove this plot from the eligible list.
			count -= 1  # Reduce number of bonuses left to place.
		if added == 0:
			print "    failed to add type",type_string,"from count",count
		if added > 0:
			print "    added",added,"of type",type_string,"and failed remaining",count
		# This bonus type is done.
		
	OutputMessage("Python: SmartMap: Step 5 Add bonuses: complete")
	checkForBadPlots()
	return 0

##########################################################################
#Step 6: place goody huts
def addGoodies():
	global width
	global height
	
	OutputMessage("Python: SmartMap: Step 6 Goody Huts")
	checkForBadPlots()
	gc = CyGlobalContext()
	cymap = CyMap()
	dice = gc.getGame().getMapRand()

	#find out the user selected terrain method
	bonusMethod = getSelectedMapValue("Bonuses:")
	print "    hutMethod (uses bonusMethod)",bonusMethod

	if bonusMethod == "standard":
		CyPythonMgr().allowDefaultImpl()
		OutputMessage("Python: SmartMap: Step 6 Goody Huts: complete standard")
		return


	#the continent lists computed in step 5
	global postContinentLists
	#goody huts
	impGoodyHut = gc.getInfoTypeForString("IMPROVEMENT_GOODY_HUT")
	players = gc.getGame().countCivPlayersEverAlive()
	plotsForGoodies = []
	
	for px in range(width):
		for py in range(height):
			if tooCloseToPole(px,py,2):
				continue
			pPlot = cymap.plot(px,py)
			#record all plots that are potential goody locations
			if pPlot.canHaveImprovement(impGoodyHut, TeamTypes.NO_TEAM, False):
				plotsForGoodies.append(pPlot)
			

	#add randomly placed goody huts, 6 per player overall, plus some extra to make them common enough
	#if you play with only a few players on a big map
	goodyHutCount = 7 * players + int(len(plotsForGoodies)/40)
	addedLocs = []
	while (goodyHutCount > 0):
		if len(plotsForGoodies) <= 0:
			break
		index = dice.get(len(plotsForGoodies), "Python: SmartMap: Dice: GoodyHut Placement")
		pPlot = plotsForGoodies[index]
		x = pPlot.getX()
		y = pPlot.getY()
		
		tooClose = False
		for usedLoc in addedLocs:
			distX,distY = getWrapDistanceXY((x,y),usedLoc)
			if distX + distY < 4:
				tooClose = true
				break
		if tooClose:
			del plotsForGoodies[index] # Remove this plot from the eligible list, it was too close to another goody
			continue
		
		
		addedLocs.append((x,y))
		pPlot.setImprovementType(impGoodyHut)
		goodyHutCount -= 1
		del plotsForGoodies[index]
		
	if goodyHutCount > 0:
		print "failed to place",goodyHutCount,"goody huts"
	OutputMessage("Python: SmartMap: Step 6 Goody Huts:complete")
	checkForBadPlots()
	return 0
		
	
##########################################################################
#Step 7: place all the players
def assignStartingPlots():
	global width
	global height
	
	OutputMessage("Python: SmartMap: Step 7 Assign starting plots")
	checkForBadPlots()
	gc = CyGlobalContext()
	cymap = CyMap()
	dice = gc.getGame().getMapRand()
	
	#find out the user selected placement method
	placeMethod = getSelectedMapValue("Start Placement:")
	print "    placeMethod",placeMethod

	if placeMethod == "standard":
		CyPythonMgr().allowDefaultImpl()
		OutputMessage("Python: SmartMap: Step 7 Assign starting plots: complete standard")
		return
	
	#the continent lists computed in step 5
	global postContinentLists
	
	#now we know how the players get distributed per continent
	#so we just have to pick the particular starting plot on each continent per player
	resourcesImportantEarly = ('BONUS_STONE', 'BONUS_MARBLE', 'BONUS_COPPER', 'BONUS_IRON', 'BONUS_HORSE', 'BONUS_FISH', 
	                           'BONUS_CRAB', 'BONUS_CLAM' ) 
	
	print "    start pre-computing plot scores"
	#pre-compute plot scores for all land plots
	plotScores = [0]*(width*height)
	baseScores = [0]*(width*height)
	for px in range(width):
		for py in range(height):
			pickedPlot = cymap.plot(px,py)
			if pickedPlot.isWater():
				continue
			pindex = getTileIndexAtXY(px,py)
			pickedScore = 0
			#the value of food etc on the actual plot tends not to matter because of the automatic
			#city minimums
			#pickedScore += 10 * int(pickedPlot.getYield(YieldTypes.YIELD_FOOD) + pickedPlot.getYield(YieldTypes.YIELD_PRODUCTION) + pickedPlot.getYield(YieldTypes.YIELD_COMMERCE))

			#other bonuses for the plot's score
			if pickedPlot.isFreshWater():
				pickedScore += 600
			if pickedPlot.isFlatlands():
				pickedScore += 200
			#a coastal start tends to be far from other players, and advantageous (can build ocean-only
			#buildings, etc)
			for nearDir in nearDirs:
				newX, newY = getTileCoordXYWithWrap((px,py), nearDir)
				if newX == px and newY == py:
					continue
				pTouchPlot = cymap.plot(newX,newY)
				if pTouchPlot.isWater() and not pTouchPlot.isLake():
					pickedScore += 1800
					break
				
			if pickedPlot.isGoody():
				pickedScore += 20
			bonusType = pickedPlot.getBonusType(-1)
			if bonusType != BonusTypes.NO_BONUS:
				pickedScore += 50
				bonusInfoXML = gc.getBonusInfo(bonusType)
				type_string = bonusInfoXML.getType()
				if type_string in resourcesImportantEarly:
					pickedScore += 50
				
			baseScores[pindex] = pickedScore

			for nearDir in playerStartCross:
				newX, newY = getTileCoordXYWithWrap((px,py), nearDir)
				if newX == px and newY == py:
					continue
				pTouchPlot = cymap.plot(newX,newY)
				pickedScore += 8 * int(pTouchPlot.getYield(YieldTypes.YIELD_FOOD) + pTouchPlot.getYield(YieldTypes.YIELD_PRODUCTION) + pTouchPlot.getYield(YieldTypes.YIELD_COMMERCE))
				if pTouchPlot.isFreshWater():
					pickedScore += 40
					if pickedPlot.isFlatlands():
						pickedScore += 40
				if pTouchPlot.isGoody():
					pickedScore += 30
				if pickedPlot.isHills():
					pickedScore += 50
				if pickedPlot.isPeak():
					pickedScore -= 80
				if pTouchPlot.isWater():
					if pTouchPlot.isLake():
						pickedScore += 40
					else:
						pickedScore -= 60
					
				bonusType = pTouchPlot.getBonusType(-1)
				if bonusType != BonusTypes.NO_BONUS:
					pickedScore += 50
					bonusInfoXML = gc.getBonusInfo(bonusType)
					type_string = bonusInfoXML.getType()
					if type_string in resourcesImportantEarly:
						pickedScore += 150
						
			plotScores[pindex] = pickedScore
	print "    finished pre-computing plot scores"
	
	#now for each continent, figure out the total score of all the land tiles it owns
	print "    score continents"
	postContinentScores = []
	for postContinentList in postContinentLists:
		totalYield = 0
		for plotPos in postContinentList:
			plotX, plotY = plotPos
			pindex = getTileIndexAtXY(plotX,plotY)
			totalYield += baseScores[pindex]
		postContinentScores.append(totalYield)
			
	print "    postContinentScores", postContinentScores
	
	players = gc.getGame().countCivPlayersEverAlive()
	
	#drop the lowest scoring continents over the min # of players because we won't use them
	while len(postContinentScores) > players:
		for index in range(len(postContinentScores)):
			if postContinentScores[index] == min(postContinentScores):
				del postContinentScores[index]
				del postContinentLists[index]
				break
		
	#pair continent lists with their scores, and sort by the scores		
	listsWithScores = []
	for index in range(len(postContinentLists)):
		listsWithScores.append( (postContinentLists[index], postContinentScores[index]) )
		
	listsWithScores = sorted(listsWithScores, key=operator.itemgetter(1))	
	listsWithScores.reverse()
	#print "    after drop listsWithScores", listsWithScores
	
	#assign initial players to continents, just round robbin, conveniently this will
	#automatically weight the early list entries higher
	playerContinentAssignments = []
	for continent in range(len(listsWithScores)):
		contList, score = listsWithScores[continent]
		playerContinentAssignments.append((score,0))
	playerList = []
	for playerNum in range(players):
		playerList.append(playerNum)
		
	playerContinent = 0
	for playerIndex in range(players):
		playerNum = playerList[playerIndex]
		score, numPlayers = playerContinentAssignments[playerContinent]
		contList, contScore = listsWithScores[playerContinent]
		score = contScore
		numPlayers += 1
		playerContinentAssignments[playerContinent] = (score, numPlayers)
		playerContinent += 1
		if playerContinent >= len(postContinentLists):
			playerContinent = 0
			
	print "    initial playerContinentAssignments",playerContinentAssignments
		
	#while any imbalance in the per-continent score divided by the number of 
	#players placed on that continent exists, rebalance players by moving them
	#from a lower scoring position to a higher scoring position
	anyImbalance = True
	while anyImbalance:
		anyImbalance = False
		#first step: find out the min/max existing scores
		minScore = 10000000
		maxScore = 0
		minScoreIndex = len(playerContinentAssignments)-1
		maxScoreIndex = 0
		for index in range(len(playerContinentAssignments)):
			score, numPlayers = playerContinentAssignments[index]
			if numPlayers > 0:
				avgScore = int(score / numPlayers)
				if avgScore < minScore:
					minScore = avgScore
					minScoreIndex = index
				if avgScore > maxScore:
					maxScore = avgScore
					maxScoreIndex = index
			
		alternativeTheory = []
		altminScore = 10000000
		altmaxScore = 0
		for index in range(len(playerContinentAssignments)):
			score, numPlayers = playerContinentAssignments[index]
			if index == maxScoreIndex:
				numPlayers += 1  
			if index == minScoreIndex:
				numPlayers -= 1  
			if numPlayers < 0:
				numPlayers = 0
				
			if numPlayers > 0:
				avgScore = int(score / numPlayers)
				if avgScore < altminScore:
					altminScore = avgScore
				if avgScore > altmaxScore:
					altmaxScore = avgScore
				
			alternativeTheory.append((score,numPlayers))
			
		print "    minScore",minScore,"maxScore",maxScore,"theory minScore",altminScore,"theory maxScore",altmaxScore
		print "    theory",alternativeTheory
		
		#if the alternative theory has better min score, or equal minscore with reduced variance (lower maxscore)
		#then we accept the alternative theory, and go through the loop again to see if we can do any better
		if altminScore > minScore or (altminScore == minScore and altmaxScore < maxScore):
			playerContinentAssignments = alternativeTheory
			anyImbalance = True
			print "    theory succeeds"
		else:
			print "    theory fails"
			
	#now we know how many players will be placed on each continent
	print "    final playerContinentAssignments",playerContinentAssignments
	
	#knowing the continents each player will start on, we must pick a starting plot on that continent
	#for the player
	finalPlots = []
	for index in range(len(playerContinentAssignments)):
		score, numPlayers = playerContinentAssignments[index]
		if numPlayers <= 0:
			continue
		print "    find placements for continent", index, "with players", numPlayers

		#keep track of the best scoring plots (the winner we'll use)
		bestScore = 0	
		bestSeparation = 0 	
		bestPickedPlotsXY = []
		#keep track of the best separated plots, to require that the 
		#finally chosen plots must come reasonably close to maximum separation
		bestMinSeparation = 0
		continentPlotList, continentScore = listsWithScores[index]
		
		print "    estimate continent separation for continent",index
		#calculate a base expected separation
		maxPossibleSeparation = 0
		for tries in range(128 * numPlayers):
			plotIndexA = dice.get(len(continentPlotList), "Python: SmartMap: Dice: continent size estimator")
			plotIndexB = dice.get(len(continentPlotList), "Python: SmartMap: Dice: continent size estimator")
			plotA = continentPlotList[plotIndexA] 
			plotB = continentPlotList[plotIndexB] 
			distX, distY = getWrapDistanceXY(plotA,plotB)
			separation = int(((distX*distX) + (distY*distY))**0.5)
			if separation > maxPossibleSeparation:
				maxPossibleSeparation = separation
					
		demandSeparation = int(75*(maxPossibleSeparation/(numPlayers+1))/100)
		print "    continent", index, "estimated initial separation", demandSeparation
		
		#we only place players on flat plots 
		allFlatPlotsCopy = []
		for plotA in continentPlotList:
			plotX,plotY = plotA
			pPlot = cymap.plot(plotX,plotY)
			if not pPlot.isFlatlands():
				continue
			allFlatPlotsCopy.append(plotA)
			
		for tries in range(64):
			pickedPlotsXY = []
			allPlots = allFlatPlotsCopy
			while len(allPlots) > (numPlayers - len(pickedPlotsXY)) and len(pickedPlotsXY) < numPlayers:
				plotIndex = dice.get(len(allPlots), "Python: SmartMap: Dice: Player Placement Plot Pick")
				plotXY = allPlots[plotIndex]
				#add this randomly picked plot to the set
				pickedPlotsXY.append(allPlots[plotIndex])
				#replace list with one with only sufficiently ranged plots
				replaceList = []
				for otherPlotXY in allPlots:
					distX,distY = getWrapDistanceXY(plotXY, otherPlotXY)
					separation = int(((distX*distX) + (distY*distY))**0.5)
					if separation > demandSeparation:
						replaceList.append(otherPlotXY)
						
				allPlots = replaceList
				#print "picked",pickedPlotsXY,"remaining",allPlots
				
			#print "found",len(pickedPlotsXY),"needed",numPlayers
			if len(pickedPlotsXY) < numPlayers:
				continue
				
			altPlotsXY = []
			for plot in pickedPlotsXY:
				givenPlotX, givenPlotY = plot
				altPlotX=-1
				altPlotY=-1
				givenPlotIndex = getTileIndexAtXY(givenPlotX,givenPlotY)
				bestNearScore = plotScores[givenPlotIndex]
				for nearDir in playerStartCross:
					altX,altY = getTileCoordXYWithWrap(plot, nearDir)
					altPlot = cymap.plot(altX,altY)
					if not altPlot.isFlatlands():
						continue
					altScoreIndex = getTileIndexAtXY(altX,altY)
					if altPlotX == -1 or altPlotY == -1 or plotScores[altScoreIndex] > bestNearScore:
						bestNearScore = plotScores[altScoreIndex]
						altPlotX = altX
						altPlotY = altY
				if altPlotX == -1 or altPlotY == -1:
					altPlotsXY.append(plot)
				else:
					altPlotsXY.append((altPlotX,altPlotY))
						
			for plotSet in [pickedPlotsXY,altPlotsXY]:
				#print "    considering plots",pickedPlotsXY 
				totalScore = 0
				for picked in plotSet:	
					plotX,plotY = picked
					pickedIndex = getTileIndexAtXY(plotX,plotY)
					totalScore += plotScores[pickedIndex]
					
				#find the minimum separation among the starting positions on this continent
				minSeparation = -1
				for firstIndex in range(len(plotSet)):
					firstPlot = plotSet[firstIndex]
					for secondIndex in range(firstIndex, len(plotSet)):
						if secondIndex != firstIndex:
							secondPlot = plotSet[secondIndex]
							distX, distY = getWrapDistanceXY(firstPlot,secondPlot)
							dist = int(((distX*distX) + (distY*distY))**0.5)
							if dist < minSeparation or minSeparation == -1:
								minSeparation = dist
					#this helps push starting positions away from positions on other continents			
					for anotherPlot in finalPlots:
						distX, distY = getWrapDistanceXY(firstPlot,anotherPlot)
						dist = int(160*((((distX*distX) + (distY*distY))**0.5))/100)
						if dist < minSeparation or minSeparation == -1:
							minSeparation = dist
							
				totalScore += minSeparation * int(bestScore / 100)
					
				#we require that the final placement have at least XX% of the best separation ever found
				if minSeparation > bestMinSeparation:
					bestMinSeparation = minSeparation
					demandSeparation = int ((75*bestMinSeparation)/100)	
					#print "        bestMinSeparation",bestMinSeparation,"demandSeparation",demandSeparation,"tries",tries,"for tiles",plotSet
				#if empty list, or better score exceeding demand separation, or old score doesn't exceed demand but this one does		
				if (bestPickedPlotsXY==[] or (totalScore > bestScore and minSeparation >= demandSeparation) 
					or (totalScore > bestScore and minSeparation >= bestSeparation)
					or  (bestSeparation < demandSeparation and minSeparation >= demandSeparation) ):
					bestScore = totalScore
					bestSeparation = minSeparation
					bestPickedPlotsXY = plotSet
					print "      new bestPickedPlotsXY",bestPickedPlotsXY,"sep",bestSeparation,"score",bestScore,"demand",demandSeparation,"numPlayers",numPlayers,"tries",tries
			
			
		print "    found placements for continent", index, "with players", numPlayers, "places",bestPickedPlotsXY
		for plotXY in bestPickedPlotsXY:
			finalPlots.append(plotXY)
			
	print "    tentative player Plots",finalPlots

	#now that we have very good plots, check nearby and make sure we didn't make a close miss
	altFinalPlots = []
	for plot in finalPlots:
		givenPlotX, givenPlotY = plot
		givenPlotIndex = getTileIndexAtXY(givenPlotX,givenPlotY)
		altPlotX = givenPlotX
		altPlotY = givenPlotY
		bestNearScore = plotScores[givenPlotIndex]
		for nearDir in playerStartCross:
			altX,altY = getTileCoordXYWithWrap(plot, nearDir)
			otherPlot = cymap.plot(altX,altY)
			if not otherPlot.isFlatlands():
				continue
			scoreIndex = getTileIndexAtXY(altX,altY)
			if plotScores[scoreIndex] > bestNearScore:
				bestNearScore = plotScores[scoreIndex]
				altPlotX = altX
				altPlotY = altY
		altFinalPlots.append((altPlotX,altPlotY))
		
	#these are the finalized plots which will be assigned to the players	
	finalPlots = altFinalPlots
	print "    final player Plots",finalPlots
	
	#maybe there was an error: we didn't find enough plots, so we'll let the standard
	#placement algorithm take care of it for us
	if len(finalPlots) < players:
		CyPythonMgr().allowDefaultImpl()
		OutputMessage("Python: SmartMap: Step 7 Assign starting plots: failed smartmap so complete standard")
		return
	
	#finally, we can actually assign the final plots to the players
	for playerIndex in range(players):
		whichStartIndex = dice.get(len(finalPlots), "Python: SmartMap: Dice: Player Placement Plot Pick Place")
		#start player 0 (human in single player) near the middle IF wrapping
		if playerIndex == 0 and (wrapX or wrapY):
			nearestMidScore = 0
			nearestMidIndex = 0
			for startIndex in range(len(finalPlots)):
				plotScore = 0
				startPlotX,startPlotY = finalPlots[startIndex]
				if wrapX:
					distFromCenterFactor = (width*width) - (abs(startPlotX-int(width/2))*abs(startPlotX-int(width/2)))
					plotScore += distFromCenterFactor
				if wrapY:
					distFromCenterFactor = (height*height) - (abs(startPlotY-int(height/2))*abs(startPlotY-int(height/2)))
					plotScore += distFromCenterFactor
				if plotScore > nearestMidScore:
					nearestMidScore = plotScore
					nearestMidIndex = startIndex
			whichStartIndex = nearestMidIndex
			
			
		#pick one of the final placement plots at random
		startX,startY = finalPlots[whichStartIndex]
		del finalPlots[whichStartIndex]
		startPlot = cymap.plot(startX,startY)
		#assign the player to it
		startPlot.setStartingPlot(True) #I believe this helps the post-processor improve map fairness
		pPlayer = gc.getPlayer(playerIndex)
		pPlayer.setStartingPlot(startPlot,True)
		playerName = pPlayer.getName()
		coastalLand = startPlot.isCoastalLand()
		print "player",playerName,"placed at",startX,startY,"coastal",coastalLand
				
	OutputMessage("Python: SmartMap: Step 7 Assign starting plots: complete")
	checkForBadPlots()
	return 0
	
	
###########################################################################
# Stuff associated with menus
###########################################################################

selection_titles = [unicode("Continents:"),
                   unicode("Ocean:"),
                   unicode("Hills:"),
                   unicode("Peaks:"),
                   unicode("Wrap:"),
                   unicode("Terrain:"),
                   unicode("Forest/Jungle:"),
                   unicode("Features:"),
                   unicode("Bonuses:"),
                   unicode("Start Placement:"),
                   unicode("Override Width:"),
                   unicode("Override Height:"),
                   unicode("Land Style:"),
                   ]
	
#dropdown list names and corresponding values used in code
selection_names_and_values = [
	                  [
	                   ["1 pangea",1],
	                   ["2 huge lands",2],
	                   ["3 large lands",3],
	                   ["4 continents",4],
	                   ["5 continents",5],
	                   ["6 continents",6],
	                   ["7 continents",7],
	                   ["8 continents",8],
	                   ["12 islands",12],
	                   ["18 isles",18],
	                   ["24 isles",24],
	                   ["36 islets",36],
	                   ["random 1-6",-1],
	                   ["random 2-5",-2],
	                   ["random 3-8",-3],
	                   ["random 4-12",-4],
	                   ["random 1-12",-6],
	                   ["1 per player",-7],
	                   ["random 2-7",-5],
	                   ],
	                  [
	                   ["minimal ocean",10],
	                   ["40% ocean",40],
	                   ["50% ocean",50],
	                   ["60% ocean",60],
	                   ["65% ocean",65],
	                   ["70% ocean",70],
	                   ["75% ocean",75],
	                   ["80% ocean",80],
	                   ["85% ocean",85],
	                   ["90% ocean",90],
	                   ["small ocean",-1],
	                   ["big ocean",-2],
	                   ["normal ocean",-3],
	                   ],
	                  [
	                   ["no hills",0],
	                   ["2% hills",2],
	                   ["4% hills",4],
	                   ["6% hills",6],
	                   ["8% hills",8],
	                   ["10% hills",10],
	                   ["12% hills",12],
	                   ["14% hills",14],
	                   ["16% hills",16],
	                   ["18% hills",18],
	                   ["20% hills",20],
	                   ["25% hills",25],
	                   ["30% hills",30],
	                   ["few hills",-1],
	                   ["many hills",-2],
	                   ["normal hills",-3],
	                   ],
	                  [
	                   ["no peaks",0],
	                   ["1% peaks",1],
	                   ["2% peaks",2],
	                   ["3% peaks",3],
	                   ["4% peaks",4],
	                   ["5% peaks",5],
	                   ["6% peaks",6],
	                   ["7% peaks",7],
	                   ["8% peaks",8],
	                   ["9% peaks",9],
	                   ["10% peaks",10],
	                   ["15% peaks",15],
	                   ["20% peaks",20],
	                   ["few peaks",-1],
	                   ["many peaks",-2],
	                   ["normal peaks",-3],
	                   ],
	                  [
	                   ["wrap y",(False,True)],
	                   ["wrap both",(True,True)],
	                   ["no wrap",(False,False)],
	                   ["wrap x",(True,False)],
	                   ],
	                  [
	                   ["standard","standard"],
	                   ["great plains","great plains"],
	                   ["oasis","oasis"],
	                   ["SmartMap","SmartMap"],
	                   ],
	                  [
	                   ["light forest,normal jungle", (-20,0)],
	                   ["light forest,heavy jungle", (-20,40)],
	                   ["normal forest,light jungle", (0,-20)],
	                   ["normal forest,heavy jungle", (0,40)],
	                   ["heavy forest,light jungle", (40,-20)],
	                   ["heavy forest,normal jungle", (40,0)],
	                   ["light forest,light jungle", (-20,-20)],
	                   ["heavy forest,heavy jungle", (40,40)],
	                   ["normal forest,normal jungle", (0,0)],
	                   ],
	                  [
	                   ["standard","standard"],
	                   ["great plains","great plains"],
	                   ["oasis","oasis"],
	                   ["SmartMap Strict","SmartMap Strict"],
	                   ["SmartMap","SmartMap"],
	                   ],
	                  [
	                   ["standard","standard"],
	                   ["SmartMap","SmartMap"],
	                   ],
	                  [
	                   ["standard","standard"],
	                   ["SmartMap","SmartMap"],
	                   ],
	                  [
	                   ["8",8],
	                   ["12",12],
	                   ["16",16],
	                   ["20",20],
	                   ["24",24],
	                   ["28",28],
	                   ["32",32],
	                   ["36",36],
	                   ["40",40],
	                   ["44",44],
	                   ["48",48],
	                   ["52",52],
	                   ["56",56],
	                   ["60",60],
	                   ["64 (could crash)",64],
	                   ["80 (could crash)",80],
	                   ["96 (could crash)",96],
	                   ["112 (could crash)",112],
	                   ["128 (could crash)",128],
	                   ["160 (could crash)",160],
	                   ["192 (could crash)",192],
	                   ["224 (could crash)",224],
	                   ["256 (could crash)",256],
	                   ["don't override",-1],
	                   ],
	                  [
	                   ["8",8],
	                   ["12",12],
	                   ["16",16],
	                   ["20",20],
	                   ["24",24],
	                   ["28",28],
	                   ["32",32],
	                   ["36",36],
	                   ["40",40],
	                   ["44",44],
	                   ["48",48],
	                   ["52",52],
	                   ["56",56],
	                   ["60",60],
	                   ["64 (could crash)",64],
	                   ["80 (could crash)",80],
	                   ["96 (could crash)",96],
	                   ["112 (could crash)",112],
	                   ["128 (could crash)",128],
	                   ["160 (could crash)",160],
	                   ["192 (could crash)",192],
	                   ["224 (could crash)",224],
	                   ["256 (could crash)",256],
	                   ["don't override",-1],
	                   ],
	                  [
	                   ["least round, extreme fragments", (-300,300)],
	                   ["least round, many fragments", (-300,100)],
	                   ["least round, some fragments", (-300,33)],
	                   ["least round, no fragments", (-300,0)],
	                   ["somewhat round, extreme fragments", (-100,300)],
	                   ["somewhat round, many fragments", (-100,100)],
	                   ["somewhat round, some fragments", (-100,33)],
	                   ["somewhat round, no fragments", (-100,0)],
	                   ["very round, extreme fragments", (0,300)],
	                   ["very round, many fragments", (0,100)],
	                   ["very round, some fragments", (0,33)],
	                   ["very round, no fragments", (0,0)],
	                   ],
	                   ]

#I don't know where this shows up!  You would think this would be
#used as the text for the Map: button, but that uses the name of
#the script instead!
def getDescription():
	#print "    getDescription"
	return unicode("SmartMap")

#return the total number of custom options for this map
def getNumCustomMapOptions():
	return len(selection_names_and_values)

#return the list of options to configure how this map is generated
def getCustomMapOptionName(argsList):
	#print "getCustomMapOptionName",argsList
	index = argsList[0]
	return selection_titles[index]

#this custom menu function returns the 'value' from a map option selected by the player
#for use by other code.
def getSelectedMapValue(optionName):
	cymap = CyMap()
	optionIndex = 0
	found = False
	for optionIndex in range(len(selection_titles)):
		title = selection_titles[optionIndex]
		if title == optionName:
			found = True
			break
			
	if not found:
		print "    !error, couldn't find entry for option named",optionName
		
	#get the user selected index (which item in the dropdown list the user selected)
	userSelectedIndex = int(cymap.getCustomMapOption(optionIndex))
	if userSelectedIndex >= len(selection_names_and_values[optionIndex]):
		userSelectedIndex = 0
	#print "get userSelectedIndex", userSelectedIndex, "optionIndex",optionIndex
	#return the corresponding integer value
	return selection_names_and_values[optionIndex][userSelectedIndex][1]
	                   
#return the number of drop down entries for each custom map option
def getNumCustomMapOptionValues(argsList):
	#print "getNumCustomMapOptionValues",argsList
	index = argsList[0]
	return len(selection_names_and_values[index])
	                   
#return the description of the custom map options (for use in the dropdown)	
def getCustomMapOptionDescAt(argsList):
	#print "getCustomMapOptionDescAt",argsList
	iOption = argsList[0]
	iSelection = argsList[1]
	return unicode(selection_names_and_values[iOption][iSelection][0])
	
#return the default drop-down entry for each custom map options
def getCustomMapOptionDefault(argsList):
	#print "getCustomMapOptionDefault",argsList
	index = argsList[0]
	#this chooses the last option in every case, which i've picked out to be a good option
	return len(selection_names_and_values[index]) - 1
	#replace with return -1 to set 'random' as the default choice
	#note that this is dangerous because it might pick an extremely large random choice of
	#width and height
	#return -1

#this is an advanced map, only shows up if you do custom map
def isAdvancedMap():
	"This map should not show up in simple mode"
	return 1

#this map uses the sea level option to determine continent separation minimum
def isSeaLevelMap():
	return 1
	
#this map may or may not obey the climate selector depending on which
#terrain and feature gen is chosen
def isClimateMap():
	return 1
	
###########################################################################
#Assorted helper functions
###########################################################################

#this helper wraps the NiTextOut and also prints to the hapdebugger
def OutputMessage(message):
	NiTextOut(message)
	print message

#before step one, we must know the wrap in order to bias the width and height
def beforeStepOne():
	global wrapX
	global wrapY
	wrapX, wrapY = getSelectedMapValue("Wrap:")
	print "    before step one wrapxy", wrapX, wrapY
	
#before step two, we must know the map width and height
def beforeStepTwo():
	cymap = CyMap()
	global width
	width = cymap.getGridWidth()
	global height
	height = cymap.getGridHeight()
	print "    before step two width", width, "height", height
	
#compute the continent lists (lists of land reachable by land travelers on each continent)
def computeContinents():
	global width
	global height
	OutputMessage("Python: SmartMap: pre-compute continents")
	cymap = CyMap()
	#first collect all continents, this isn't the same as the continent lists
	#calculated during land layout because post generation touchups may have
	#changed things
	print "    start continent analysis"
	global postContinentLists
	checkForBadPlots()
	postContinentLists = []
	plotMarks = [0] * (width*height)
	plotMark = 0
	processPlots = []
	for x in range(width):
		for y in range(height):
			if tooCloseToPole(x,y,2):
				continue
				
			index = getTileIndexAtXY(x,y)
			if plotMarks[index] != 0:
				continue
			#if not an interesting plot, continue
			pPlot = cymap.plot(x,y)
			if not pPlot.isFlatlands() and not pPlot.isHills():
				continue
			postContinentList = []
			processPlots.append((x,y))
			plotMark += 1
			postContinentLists.append(postContinentList)
			
			#recursive neighbor plot addition	
			while (len(processPlots) > 0):
				plotX,plotY = processPlots.pop(0)
				existingIndex = getTileIndexAtXY(plotX,plotY)
				if plotMarks[existingIndex] == plotMark:
					continue
				plotMarks[existingIndex] = plotMark
				postContinentList.append((plotX,plotY))
				for nearDir in nearDirs:
					newX, newY = getTileCoordXYWithWrap((plotX,plotY), nearDir)
					newIndex = getTileIndexAtXY(newX, newY)
					if plotMarks[newIndex] != 0:
						continue

					pProcPlot = cymap.plot(newX,newY)
					if not pProcPlot.isFlatlands() and not pProcPlot.isHills():
						continue
						
					processPlots.append((newX,newY))
					#print "    processPlots",processPlots
					
	#remove any too-short continents from the list
	anyTooShort = True
	while anyTooShort:
		anyTooShort = False
		for listIndex in range(len(postContinentLists)):
			if len(postContinentLists[listIndex]) < 10:
				del postContinentLists[listIndex]
				anyTooShort = True
				break
					
#override this method used to compute the altitude of a plot, used to determine
#river paths 
def getRiverAltitude(argsList):
	global preComputedAltitudes
	pPlot = argsList[0]
	px = pPlot.getX()
	py = pPlot.getY()
	plotIndex = getTileIndexAtXY(px,py)
	return preComputedAltitudes[plotIndex]
	
#this makes a one-time computation of the altitudes of all plots					
def preComputeAltitudes():
	global preComputedAltitudes
	global width
	global height
	OutputMessage("Python: SmartMap: pre-compute tile altitudes")
	cymap = CyMap()
	preComputedAltitudes = [0]*(width*height)
	maxAltitude = 0
	for x in range(width):
		for y in range(height):
			plotIndex = getTileIndexAtXY(x,y)
			pPlot = cymap.plot(x,y)
			altitude = 0
			if pPlot.isLake():
				altitude += 10
			#ocean is always altitude 0
			elif pPlot.isWater():
				preComputedAltitudes[plotIndex] = 0
				continue
			plotType = pPlot.getPlotType()	
			if plotType == PlotTypes.PLOT_LAND:
				altitude += 100
			if plotType == PlotTypes.PLOT_HILLS:
				altitude += 300
			if plotType == PlotTypes.PLOT_PEAK:
				altitude += 700
			for nearDir in playerStartCross:
				newX, newY = getTileCoordXYWithWrap((x,y), nearDir)
				otherPlot = cymap.plot(newX,newY)
				otherPlotType = otherPlot.getPlotType()	
				if otherPlot.isLake():
					altitude += 2
				elif otherPlot.isWater():
					altitude -= 20
				if otherPlotType == PlotTypes.PLOT_LAND:
					altitude += 5
				if otherPlotType == PlotTypes.PLOT_HILLS:
					altitude += 20
				if otherPlotType == PlotTypes.PLOT_PEAK:
					altitude += 50
			preComputedAltitudes[plotIndex] = altitude
			if altitude > maxAltitude:
				maxAltitude = altitude
	#scale all altitudes to 0..250 range			
	for x in range(width):
		for y in range(height):
			plotIndex = getTileIndexAtXY(x,y)
			preComputedAltitudes[plotIndex] = int((250*preComputedAltitudes[plotIndex])/maxAltitude)

#this makes a one-time computation of the wetness of all plots					
def preComputeWetness():
	global preComputedWetness
	global width
	global height
	OutputMessage("Python: SmartMap: pre-compute tile altitudes")
	cymap = CyMap()
	preComputedWetness = [0]*(width*height)
	maxWetness = 0
	for x in range(width):
		for y in range(height):
			plotIndex = getTileIndexAtXY(x,y)
			pPlot = cymap.plot(x,y)
			wetness = 0
			if pPlot.isFreshWater() or pPlot.isRiver():
				wetness += 96
			
			#improved wetness calcualtor
			if pPlot.isWOfRiver():
				wetness += 32
			if pPlot.isNOfRiver():
				wetness += 32
			northX,northY = getTileCoordXYWithWrap((x,y), (0,1))
			pNorthPlot = cymap.plot(northX,northY)
			if pNorthPlot.isNOfRiver():
				wetness += 32
			westX,westY = getTileCoordXYWithWrap((x,y), (-1,0))
			pWestPlot = cymap.plot(northX,northY)
			if pWestPlot.isWOfRiver():
				wetness += 32
				
			for nearDir in nearDirs:
				otherX, otherY = getTileCoordXYWithWrap((x,y), nearDir)
				otherPlot = cymap.plot(otherX,otherY)
				if otherPlot.isLake():
					wetness += 32
				if otherPlot.isFreshWater() or otherPlot.isRiver():
					wetness += 16
				if otherPlot.isWOfRiver():
					wetness += 12
				if otherPlot.isNOfRiver():
					wetness += 12
				northX,northY = getTileCoordXYWithWrap((otherX,otherY), (0,1))
				pNorthPlot = cymap.plot(northX,northY)
				if pNorthPlot.isNOfRiver():
					wetness += 12
				westX,westY = getTileCoordXYWithWrap((otherX,otherY), (-1,0))
				pWestPlot = cymap.plot(northX,northY)
				if pWestPlot.isWOfRiver():
					wetness += 12
				
			for nearDir in playerStartCross:
				if nearDir in nearDirs:
					continue
				otherX, otherY = getTileCoordXYWithWrap((x,y), nearDir)
				otherPlot = cymap.plot(otherX,otherY)
				if otherPlot.isLake():
					wetness += 12
				if otherPlot.isFreshWater() or otherPlot.isRiver():
					wetness += 4
				if otherPlot.isWOfRiver():
					wetness += 4
				if otherPlot.isNOfRiver():
					wetness += 4
				northX,northY = getTileCoordXYWithWrap((otherX,otherY), (0,1))
				pNorthPlot = cymap.plot(northX,northY)
				if pNorthPlot.isNOfRiver():
					wetness += 4
				westX,westY = getTileCoordXYWithWrap((otherX,otherY), (-1,0))
				pWestPlot = cymap.plot(northX,northY)
				if pWestPlot.isWOfRiver():
					wetness += 4

			preComputedWetness[plotIndex] = wetness
			if wetness > maxWetness:
				maxWetness = wetness
	#scale all wetness to 0..250 range			
	for x in range(width):
		for y in range(height):
			plotIndex = getTileIndexAtXY(x,y)
			preComputedWetness[plotIndex] = int((250*preComputedWetness[plotIndex])/maxWetness)

	
#this helper function returns the tile coordinate in a given direction,
# accounting for xy wrapping, note that this method may bounce back from boundaries
def getTileCoordXYWithWrap(position,direction):
	global wrapX
	global wrapY
	global width
	global height
	x,y = position
	dx,dy = direction
	x += dx
	y += dy
	#print "    wrapx",wrapX,"wrapY",wrapY,"True",True,"False",False
	if not wrapX and x >= width-3:
		x = width-3
	elif not wrapX and x <= 2:
		x = 2
	elif wrapX and x > width-1:
		x = x-width
	elif wrapX and x < 0:
		x = x+width
	if not wrapY and y >= height-3:
		y = height-3
	if not wrapY and y <= 2:
		y = 2
	elif wrapY and y > height-1:
		y = y-height
	elif wrapY and y < 0:
		y = y+height
	if x < 0:
		x = 0
	elif x > width-1:
		x = width-1
	if y < 0:
		y = 0
	elif y > height-1:
		y = height-1
	return (x,y)
	
#this helper function converts an x,y position to a tile index
#this helps to avoid ever making a mistake in computing this consistently
def getTileIndexAtXY(x,y):
	global width
	return ((y*width) + x)
	
#get the distance between two coordinates, considering that the coordinate
#system may wrap in either x or y
def getWrapDistanceXY(posA,posB):
	global wrapX
	global wrapY
	global width
	global height
	ax,ay = posA
	bx,by = posB
	xDiff = abs(ax-bx)
	yDiff = abs(ay-by)
		
	if wrapX:
		altXDiff = abs(ax-bx+width)
		if altXDiff < xDiff:
			xDiff = altXDiff
		altXDiff = abs(ax-bx-width)
		if altXDiff < xDiff:
			xDiff = altXDiff
	
	if wrapY:
		altYDiff = abs(ay-by+height)
		if altYDiff < yDiff:
			yDiff = altYDiff
		altYDiff = abs(ay-by-height)
		if altYDiff < yDiff:
			yDiff = altYDiff
			
	return (xDiff, yDiff)
	
#return true if a given point is within range coords of a pole
def tooCloseToPole(x,y,range):
	global wrapX
	global wrapY
	if (not wrapX and (x<=range or x>=width-(range+1))):
		return True
	if (not wrapY and (y<=range or y>=height-(range+1))):
		return True
	return False
	
#somewhere in the default methods, an occasional bad tile is placed
#and this method corrects for that
def checkForBadPlots():
	global wrapX
	global wrapY
	global width
	global height

	OutputMessage("Python: SmartMap: Check For Bad Plots")
	gc = CyGlobalContext()
	cymap = CyMap()
	featIce = gc.getInfoTypeForString("FEATURE_ICE")

	#damn, where is this happening!
	for x in range(width):
		for y in range(height):
			pPlot = cymap.plot(x,y)
			plotType = pPlot.getPlotType()
			if tooCloseToPole(x,y,1):
				if plotType != PlotTypes.PLOT_OCEAN:
					print "bad xy",x,y,plotType
				pPlot.setPlotType(PlotTypes.PLOT_OCEAN,True,True)
				pPlot.setFeatureType(featIce,-1)
				pPlot.setBonusType(-1)
	
#this helper function will tell you what % the plot is from a pole	
def percentFromPole(x,y):
	global wrapX
	global wrapY
	global width
	global height
	halfX = int((width-1)/2)
	halfY = int((height-1)/2)
	
	xPercentFromPole = 100
	if not wrapX:
		xFromPole = x
		if x > halfX:
			xFromPole = (width-1) - x
		xPercentFromPole = int((xFromPole*100)/halfX)

	yPercentFromPole = 100
	if not wrapY:
		yFromPole = y
		if y > halfY:
			yFromPole = (height-1) - y
		yPercentFromPole = int((yFromPole*100)/halfY)
		
	return min(xPercentFromPole, yPercentFromPole)
	
#this helper function will tell you what % the plot is from the equator	
def percentFromEquator(x,y):
	global wrapX
	global wrapY
	global width
	global height
	halfX = int((width-1)/2)
	halfY = int((height-1)/2)
	
	xPercentFromEquator = 100
	if not wrapX:
		xFromEquator = halfX - x
		if x > halfX:
			xFromEquator = x - halfX
		xPercentFromEquator = int((xFromEquator*100)/halfX)

	yPercentFromEquator = 100
	if not wrapY:
		yFromEquator = halfY - y
		if y > halfY:
			yFromEquator = y - halfY
		yPercentFromEquator = int((yFromEquator*100)/halfY)
		
	return min(xPercentFromEquator, yPercentFromEquator)
	
		

##########################################################################
##########################################################################
##########################################################################
##########################################################################
#code below here is ripped from the standard map scripts, to allow the selection
#of other script behaviors on some of the options
#some needed classes
# subclass TerrainGenerator to redefine everything. This is a regional map.
class GreatPlainsTerrainGenerator(CvMapGeneratorUtil.TerrainGenerator):
	def __init__(self, iRockyDesertPercent=50, iRockyPlainsPercent=30, 
	             iGrassPercent=17, iDesertPercent=8, iTexDesertPercent=20,
	             iEastDesertPercent=2, iEastPlainsPercent=23, 
	             fWestLongitude=0.15, fEastLongitude=0.65, fTexLat=0.37,
	             fTexEast=0.55, fracXExp=-1, fracYExp=-1, grain_amount=4):
		 #Note: If you change longitude values here, then you will...
		 #...need to change them elsewhere in the script, as well.
		self.gc = CyGlobalContext()
		self.map = CyMap()

		self.grain_amount = grain_amount + self.gc.getWorldInfo(self.map.getWorldSize()).getTerrainGrainChange()

		self.iWidth = self.map.getGridWidth()
		self.iHeight = self.map.getGridHeight()

		self.mapRand = self.gc.getGame().getMapRand()

		self.iFlags = 0  # Disallow FRAC_POLAR flag, to prevent "zero row" problems.

		self.rocky=CyFractal()
		self.plains=CyFractal()
		self.east=CyFractal()
		self.variation=CyFractal()

		self.iRockyDTopPercent = 100
		self.iRockyDBottomPercent = max(0,int(100-iRockyDesertPercent))
		self.iRockyPTopPercent = int(100 - iRockyDesertPercent)
		self.iRockyPBottomPercent = max(0,int(100-iRockyDesertPercent-iRockyPlainsPercent))
		self.iDesertTopPercent = 100
		self.iDesertBottomPercent = max(0,int(100-iDesertPercent))
		self.iTexDesertTopPercent = 70
		self.iTexDesertBottomPercent = max(0,int(70-iTexDesertPercent))
		self.iGrassTopPercent = int(iGrassPercent)
		self.iGrassBottomPercent = 0
		self.iTexGrassBottomPercent = 6
		self.iEastDTopPercent = 100
		self.iEastDBottomPercent = max(0,int(100-iEastDesertPercent))
		self.iEastPTopPercent = int(100 - iEastDesertPercent)
		self.iEastPBottomPercent = max(0,int(100-iEastDesertPercent-iEastPlainsPercent))
		self.iMountainTopPercent = 75
		self.iMountainBottomPercent = 60

		self.fWestLongitude = fWestLongitude
		self.fEastLongitude = fEastLongitude
		self.fTexLat = fTexLat
		self.fTexEast = fTexEast

		self.iGrassPercent = iGrassPercent
		self.iDesertPercent = iDesertPercent
		self.iTexDesertPercent = iTexDesertPercent
		self.iEastDesertPercent = iEastDesertPercent
		self.iEastPlainsPercent = iEastPlainsPercent
		self.iRockyDesertPercent = iRockyDesertPercent
		self.iRockyPlainsPercent = iRockyPlainsPercent

		self.fracXExp = fracXExp
		self.fracYExp = fracYExp

		self.initFractals()
		
	def initFractals(self):
		self.rocky.fracInit(self.iWidth, self.iHeight, self.grain_amount, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iRockyDTop = self.rocky.getHeightFromPercent(self.iRockyDTopPercent)
		self.iRockyDBottom = self.rocky.getHeightFromPercent(self.iRockyDBottomPercent)
		self.iRockyPTop = self.rocky.getHeightFromPercent(self.iRockyPTopPercent)
		self.iRockyPBottom = self.rocky.getHeightFromPercent(self.iRockyPBottomPercent)

		self.plains.fracInit(self.iWidth, self.iHeight, self.grain_amount+1, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iDesertTop = self.plains.getHeightFromPercent(self.iDesertTopPercent)
		self.iDesertBottom = self.plains.getHeightFromPercent(self.iDesertBottomPercent)
		self.iTexDesertTop = self.plains.getHeightFromPercent(self.iTexDesertTopPercent)
		self.iTexDesertBottom = self.plains.getHeightFromPercent(self.iTexDesertBottomPercent)
		self.iGrassTop = self.plains.getHeightFromPercent(self.iGrassTopPercent)
		self.iGrassBottom = self.plains.getHeightFromPercent(self.iGrassBottomPercent)
		self.iTexGrassBottom = self.plains.getHeightFromPercent(self.iTexGrassBottomPercent)

		self.east.fracInit(self.iWidth, self.iHeight, self.grain_amount, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iEastDTop = self.east.getHeightFromPercent(self.iEastDTopPercent)
		self.iEastDBottom = self.east.getHeightFromPercent(self.iEastDBottomPercent)
		self.iEastPTop = self.east.getHeightFromPercent(self.iEastPTopPercent)
		self.iEastPBottom = self.east.getHeightFromPercent(self.iEastPBottomPercent)

		self.variation.fracInit(self.iWidth, self.iHeight, self.grain_amount, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)

		self.terrainDesert = self.gc.getInfoTypeForString("TERRAIN_DESERT")
		self.terrainPlains = self.gc.getInfoTypeForString("TERRAIN_PLAINS")
		self.terrainGrass = self.gc.getInfoTypeForString("TERRAIN_GRASS")

	def getLatitudeAtPlot(self, iX, iY):
		lat = iX/float(self.iWidth) # 0.0 = west

		# Adjust latitude using self.variation fractal, to mix things up:
		lat += (128 - self.variation.getHeight(iX, iY))/(255.0 * 5.0)

		# Limit to the range [0, 1]:
		if lat < 0:
			lat = 0.0
		if lat > 1:
			lat = 1.0

		return lat

	def generateTerrain(self):		
		terrainData = [0]*(self.iWidth*self.iHeight)
		for x in range(self.iWidth):
			for y in range(self.iHeight):
				iI = y*self.iWidth + x
				terrain = self.generateTerrainAtPlot(x, y)
				terrainData[iI] = terrain
		return terrainData

	def generateTerrainAtPlot(self,iX,iY):
		lat = self.getLatitudeAtPlot(iX,iY)

		if (self.map.plot(iX, iY).isWater()):
			return self.map.plot(iX, iY).getTerrainType()

		if lat <= self.fWestLongitude:
			val = self.rocky.getHeight(iX, iY)
			if val >= self.iRockyDBottom and val <= self.iRockyDTop:
				terrainVal = self.terrainDesert
			elif val >= self.iRockyPBottom and val <= self.iRockyPTop:
				terrainVal = self.terrainPlains
			else:
				long = iY/float(self.iHeight)
				if long > 0.23:
					terrainVal = self.terrainGrass
				else:
					terrainVal = self.terrainDesert
		elif lat > self.fEastLongitude:
			val = self.east.getHeight(iX, iY)
			if val >= self.iEastDBottom and val <= self.iEastDTop:
				terrainVal = self.terrainDesert
			elif val >= self.iEastPBottom and val <= self.iEastPTop:
				terrainVal = self.terrainPlains
			else:
				terrainVal = self.terrainGrass
		elif lat > self.fWestLongitude and lat <= self.fTexEast and iY/float(self.iHeight) <= self.fTexLat:
			# More desert added to Texas region at Soren's request.
			val = self.east.getHeight(iX, iY)
			if val >= self.iDesertBottom and val <= self.iDesertTop:
				terrainVal = self.terrainDesert
			elif val >= self.iTexDesertBottom and val <= self.iTexDesertTop:
				terrainVal = self.terrainDesert
			elif val >= self.iTexGrassBottom and val <= self.iGrassTop:
				terrainVal = self.terrainGrass
			else:
				terrainVal = self.terrainPlains
		else:
			val = self.plains.getHeight(iX, iY)
			if val >= self.iDesertBottom and val <= self.iDesertTop:
				terrainVal = self.terrainDesert
			elif val >= self.iGrassBottom and val <= self.iGrassTop:
				terrainVal = self.terrainGrass
			else:
				terrainVal = self.terrainPlains

		if (terrainVal == TerrainTypes.NO_TERRAIN):
			return self.map.plot(iX, iY).getTerrainType()

		return terrainVal


# subclass TerrainGenerator to redefine everything. This is a regional map. Ice need not apply!
# Latitudes, ratios, the works... It's all rewired. - Sirian June 20, 2005

class OasisTerrainGenerator(CvMapGeneratorUtil.TerrainGenerator):
	def __init__(self, iGrassPercent=50, iPlainsPercent=35,
	             iNorthernPlainsPercent=40, iOasisGrassPercent=9,
	             iOasisPlainsPercent=16, iOasisTopLatitude=0.69,
	             iJungleLatitude=0.14, iOasisBottomLatitude=0.3,
	             fracXExp=-1, fracYExp=-1, grain_amount=4):
		
		self.grain_amount = grain_amount
		
		self.gc = CyGlobalContext()
		self.map = CyMap()

		self.iWidth = self.map.getGridWidth()
		self.iHeight = self.map.getGridHeight()

		self.mapRand = self.gc.getGame().getMapRand()
		self.iFlags = 0

		self.grass=CyFractal()
		self.Oasisgrass=CyFractal()
		self.plains=CyFractal()
		self.Oasisplains=CyFractal()
		self.northernplains=CyFractal()
		self.variation=CyFractal()

		self.iGrassTopPercent = 100
		self.iGrassBottomPercent = max(0,int(100-iGrassPercent))
		self.iPlainsTopPercent = 100
		self.iPlainsBottomPercent = max(0,int(100-iGrassPercent-iPlainsPercent))
		self.iOasisGrassTopPercent = 100
		self.iOasisGrassBottomPercent = max(0,int(100-iOasisGrassPercent))
		self.iOasisPlainsTopPercent = 100
		self.iOasisPlainsBottomPercent = max(0,int(100-iOasisGrassPercent-iOasisPlainsPercent))
		self.iNorthernPlainsBottomPercent = max(0,int(100-iNorthernPlainsPercent))
		self.iMountainTopPercent = 75
		self.iMountainBottomPercent = 60
		
		self.iOasisBottomLatitude = iOasisBottomLatitude
		self.iOasisTopLatitude = iOasisTopLatitude
		self.iJungleLatitude = iJungleLatitude
		
		self.iGrassPercent = iGrassPercent
		self.iPlainsPercent = iPlainsPercent
		self.iOasisGrassPercent = iOasisGrassPercent
		self.iOasisPlainsPercent = iOasisPlainsPercent
		self.iNorthernPlainsPercent = iNorthernPlainsPercent
		
		self.fracXExp = fracXExp
		self.fracYExp = fracYExp

		self.initFractals()
		
	def initFractals(self):
		self.grass.fracInit(self.iWidth, self.iHeight, self.grain_amount, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iGrassTop = self.grass.getHeightFromPercent(self.iGrassTopPercent)
		self.iGrassBottom = self.grass.getHeightFromPercent(self.iGrassBottomPercent)

		self.plains.fracInit(self.iWidth, self.iHeight, self.grain_amount+1, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iPlainsTop = self.plains.getHeightFromPercent(self.iPlainsTopPercent)
		self.iPlainsBottom = self.plains.getHeightFromPercent(self.iPlainsBottomPercent)

		self.Oasisgrass.fracInit(self.iWidth, self.iHeight, self.grain_amount, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iOasisGrassTop = self.grass.getHeightFromPercent(self.iOasisGrassTopPercent)
		self.iOasisGrassBottom = self.grass.getHeightFromPercent(self.iOasisGrassBottomPercent)

		self.Oasisplains.fracInit(self.iWidth, self.iHeight, self.grain_amount+1, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iOasisPlainsTop = self.plains.getHeightFromPercent(self.iOasisPlainsTopPercent)
		self.iOasisPlainsBottom = self.plains.getHeightFromPercent(self.iOasisPlainsBottomPercent)

		self.northernplains.fracInit(self.iWidth, self.iHeight, self.grain_amount+1, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.iNorthernPlainsBottom = self.plains.getHeightFromPercent(self.iNorthernPlainsBottomPercent)

		self.variation.fracInit(self.iWidth, self.iHeight, self.grain_amount, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)

		self.terrainDesert = self.gc.getInfoTypeForString("TERRAIN_DESERT")
		self.terrainPlains = self.gc.getInfoTypeForString("TERRAIN_PLAINS")
		self.terrainGrass = self.gc.getInfoTypeForString("TERRAIN_GRASS")

	def getLatitudeAtPlot(self, iX, iY):
		lat = iY/float(self.iHeight) # 0.0 = south edge, 1.0 = north edge

		 #Adjust latitude using self.variation fractal, to mix things up:
		lat += (128 - self.variation.getHeight(iX, iY))/(255.0 * 5.0)

		# Limit to the range [0, 1]:
		if lat < 0:
			lat = 0.0
		if lat > 1:
			lat = 1.0

		return lat

	def generateTerrain(self):		
		terrainData = [0]*(self.iWidth*self.iHeight)
		for x in range(self.iWidth):
			for y in range(self.iHeight):
				iI = y*self.iWidth + x
				terrain = self.generateTerrainAtPlot(x, y)
				terrainData[iI] = terrain
		return terrainData

	def generateTerrainAtPlot(self,iX,iY):
		lat = self.getLatitudeAtPlot(iX,iY)

		if (self.map.plot(iX, iY).isWater()):
			return self.map.plot(iX, iY).getTerrainType()

		terrainVal = self.terrainDesert

		if lat > self.iOasisTopLatitude:
			plainsVal = self.plains.getHeight(iX, iY)
			if plainsVal >= self.iNorthernPlainsBottom:
				terrainVal = self.terrainPlains
			else:
				terrainVal = self.terrainGrass
		elif lat < self.iJungleLatitude:
			terrainVal = self.terrainGrass
		elif lat < self.iOasisBottomLatitude and lat >= self.iJungleLatitude:
			grassVal = self.grass.getHeight(iX, iY)
			plainsVal = self.plains.getHeight(iX, iY)
			if ((grassVal >= self.iGrassBottom) and (grassVal <= self.iGrassTop)):
				terrainVal = self.terrainGrass
			elif ((plainsVal >= self.iPlainsBottom) and (plainsVal <= self.iPlainsTop)):
				terrainVal = self.terrainPlains
		else:
			OasisgrassVal = self.grass.getHeight(iX, iY)
			OasisplainsVal = self.plains.getHeight(iX, iY)
			if ((OasisgrassVal >= self.iOasisGrassBottom) and (OasisgrassVal <= self.iOasisGrassTop)):
				terrainVal = self.terrainGrass
			elif ((OasisplainsVal >= self.iOasisPlainsBottom) and (OasisplainsVal <= self.iOasisPlainsTop)):
				terrainVal = self.terrainPlains

		if (terrainVal == TerrainTypes.NO_TERRAIN):
			return self.map.plot(iX, iY).getTerrainType()

		return terrainVal

class GreatPlainsFeatureGenerator(CvMapGeneratorUtil.FeatureGenerator):
	def __init__(self, iJunglePercent=40, iEastForestPercent=45, 
	             iForestPercent=8, iRockyForestPercent=55, 
	             forest_grain=6, fracXExp=-1, fracYExp=-1):
		self.gc = CyGlobalContext()
		self.map = CyMap()
		self.mapRand = self.gc.getGame().getMapRand()
		self.forests = CyFractal()
		
		self.iFlags = 0  # Disallow FRAC_POLAR flag, to prevent "zero row" problems.

		self.iGridW = self.map.getGridWidth()
		self.iGridH = self.map.getGridHeight()
		
		self.iJunglePercent = iJunglePercent
		self.iForestPercent = iForestPercent
		self.iEastForestPercent = iEastForestPercent
		self.iRockyForestPercent = iRockyForestPercent
		
		self.forest_grain = forest_grain + self.gc.getWorldInfo(self.map.getWorldSize()).getFeatureGrainChange()

		self.fracXExp = fracXExp
		self.fracYExp = fracYExp

		self.__initFractals()
		self.__initFeatureTypes()
	
	def __initFractals(self):
		self.forests.fracInit(self.iGridW, self.iGridH, self.forest_grain, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		
		self.iJungleLevel = self.forests.getHeightFromPercent(100 - self.iJunglePercent)
		self.iForestLevel = self.forests.getHeightFromPercent(self.iForestPercent)
		self.iEastForestLevel = self.forests.getHeightFromPercent(self.iEastForestPercent)
		self.iRockyForestLevel = self.forests.getHeightFromPercent(self.iRockyForestPercent)
		
	def __initFeatureTypes(self):
		self.featureJungle = self.gc.getInfoTypeForString("FEATURE_JUNGLE")
		self.featureForest = self.gc.getInfoTypeForString("FEATURE_FOREST")
		self.featureOasis = self.gc.getInfoTypeForString("FEATURE_OASIS")

	def getLatitudeAtPlot(self, iX, iY):
		 #0.0 = bottom edge, 1.0 = top edge.
		return iX/float(self.iGridW)

	def addFeaturesAtPlot(self, iX, iY):
		#"adds any appropriate features at the plot (iX, iY) where (0,0) is in the SW"
		long = iX/float(self.iGridW)
		lat = iY/float(self.iGridH)
		pPlot = self.map.sPlot(iX, iY)

		for iI in range(self.gc.getNumFeatureInfos()):
			if pPlot.canHaveFeature(iI):
				if self.mapRand.get(10000, "Add Feature PYTHON") < self.gc.getFeatureInfo(iI).getAppearanceProbability():
					pPlot.setFeatureType(iI, -1)

		if (pPlot.getFeatureType() == FeatureTypes.NO_FEATURE):
			# Jungles only in Louisiana Wetlands!
			if long > 0.65 and lat < 0.45:
				self.addJunglesAtPlot(pPlot, iX, iY, lat)
			
		if (pPlot.getFeatureType() == FeatureTypes.NO_FEATURE):
			self.addForestsAtPlot(pPlot, iX, iY, lat, long)
		
	def addIceAtPlot(self, pPlot, iX, iY, lat):
		 #We don' need no steeking ice. M'kay? Alrighty then.
		ice = 0
	
	def addJunglesAtPlot(self, pPlot, iX, iY, lat):
		 #Warning: this version of JunglesAtPlot is using the forest fractal!
		if pPlot.canHaveFeature(self.featureJungle):
			if (self.forests.getHeight(iX, iY) >= self.iJungleLevel):
				pPlot.setFeatureType(self.featureJungle, -1)

	def addForestsAtPlot(self, pPlot, iX, iY, lat, long):
		 #Evergreens in the Rockies. (Pinion Pines! Juniper! Spruce!)
		 #Deciduous trees elsewhere.
		if (long < 0.16 and lat > 0.23) and (pPlot.isFlatlands() or pPlot.isHills()):
			if self.forests.getHeight(iX, iY) <= self.iRockyForestLevel:
				pPlot.setFeatureType(self.featureForest, 2)
		elif long > 0.72 and pPlot.canHaveFeature(self.featureForest):
			if self.forests.getHeight(iX, iY) <= self.iEastForestLevel:
				pPlot.setFeatureType(self.featureForest, 0)
		else:
			if pPlot.canHaveFeature(self.featureForest):
				if self.forests.getHeight(iX, iY) <= self.iForestLevel:
					pPlot.setFeatureType(self.featureForest, 0)

class OasisFeatureGenerator(CvMapGeneratorUtil.FeatureGenerator):
	def __init__(self, iJunglePercent=40, iForestPercent=45,
                     jungle_grain=5, forest_grain=6, fracXExp=-1, fracYExp=-1):
		self.gc = CyGlobalContext()
		self.map = CyMap()
		self.mapRand = self.gc.getGame().getMapRand()
		self.jungles = CyFractal()
		self.forests = CyFractal()
		self.iFlags = 0
		self.iGridW = self.map.getGridWidth()
		self.iGridH = self.map.getGridHeight()
		
		self.iJunglePercent = iJunglePercent
		self.iForestPercent = iForestPercent
		
		self.jungle_grain = jungle_grain
		self.forest_grain = forest_grain

		self.fracXExp = fracXExp
		self.fracYExp = fracYExp

		self.__initFractals()
		self.__initFeatureTypes()
	
	def __initFractals(self):
		self.jungles.fracInit(self.iGridW+1, self.iGridH+1, self.jungle_grain, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		self.forests.fracInit(self.iGridW+1, self.iGridH+1, self.forest_grain, self.mapRand, self.iFlags, self.fracXExp, self.fracYExp)
		
		self.iJungleLevel = self.jungles.getHeightFromPercent(self.iJunglePercent)
		self.iForestLevel = self.forests.getHeightFromPercent(self.iForestPercent)
		
	def __initFeatureTypes(self):
		self.featureJungle = self.gc.getInfoTypeForString("FEATURE_JUNGLE")
		self.featureForest = self.gc.getInfoTypeForString("FEATURE_FOREST")
		self.featureOasis = self.gc.getInfoTypeForString("FEATURE_OASIS")

	def getLatitudeAtPlot(self, iX, iY):
		 #0.0 = bottom edge, 1.0 = top edge.
		return iY/float(self.iGridH)

	def addFeaturesAtPlot(self, iX, iY):
		#"adds any appropriate features at the plot (iX, iY) where (0,0) is in the SW"
		lat = self.getLatitudeAtPlot(iX, iY)
		pPlot = self.map.sPlot(iX, iY)

		for iI in range(self.gc.getNumFeatureInfos()):
			if pPlot.canHaveFeature(iI):
				if self.mapRand.get(10000, "Add Feature PYTHON") < self.gc.getFeatureInfo(iI).getAppearanceProbability():
					pPlot.setFeatureType(iI, -1)

		if (pPlot.getFeatureType() == FeatureTypes.NO_FEATURE):
			# Jungles only in the deep south or in the Oasis!
			if lat < 0.16:
				self.addJunglesAtPlot(pPlot, iX, iY, lat)
			elif lat > 0.32 and lat < 0.65 and (pPlot.getTerrainType() == self.gc.getInfoTypeForString("TERRAIN_GRASS")):
				pPlot.setFeatureType(self.featureJungle, -1)

		if (pPlot.getFeatureType() == FeatureTypes.NO_FEATURE):
			# No forests in the Oasis!
			if lat > 0.71 or lat < 0.3:
				self.addForestsAtPlot(pPlot, iX, iY, lat)

		if pPlot.isFlatlands() and (pPlot.getFeatureType() == FeatureTypes.NO_FEATURE) and (pPlot.getTerrainType() == self.gc.getInfoTypeForString("TERRAIN_DESERT")):
			# Add more Oases!
			if (lat < 0.71 and lat > 0.3) and self.mapRand.get(9, "Add Extra Oases PYTHON") == 0:
				pPlot.setFeatureType(self.featureOasis, -1)
                                		
	def addIceAtPlot(self, pPlot, iX, iY, lat):
		 #We don' need no steeking ice. M'kay? Alrighty then.
		ice = 0
	
	def addJunglesAtPlot(self, pPlot, iX, iY, lat):
		if pPlot.canHaveFeature(self.featureJungle):
			if self.jungles.getHeight(iX+1, iY+1) <= self.iJungleLevel:
				pPlot.setFeatureType(self.featureJungle, -1)

	def addForestsAtPlot(self, pPlot, iX, iY, lat):
		 #No evergreens.
		if pPlot.canHaveFeature(self.featureForest):
			if self.forests.getHeight(iX+1, iY+1) <= self.iForestLevel:
				pPlot.setFeatureType(self.featureForest, 0)
