#
#	FILE:	 Fair_Continents.py
#	VERSION: 2.1
#	AUTHOR:  Doug McCreary (surt@uga.edu)
#	PURPOSE: Create some pretty fair continents
#            and demonstrate non-fractal map generation
#-----------------------------------------------------------------------------
#	Copyright (c) 2005-2006 Doug's Software All rights reserved.
#-----------------------------------------------------------------------------
#


# Version History
# 3.0 added feature generation demonstration
# 2.4 reduce strategic resource density compared to common resources
# 2.3 fix problem where combination of choosing a small map and a high width/height
#     would result in too much ocean and unplayable maps
# 2.2 fixed oops causing the horizontal continent count to be ignored
# 2.1 changed algorithm for determining where ocean is placed to make
#     the continent sizes fairer, improved documentation
# 2.0 added custom terrain generation, bonus placement, sea level support
# 1.0 (as Surt.py) support for laying out continents, basic map function demonstration
from CvPythonExtensions import *
import CvUtil
import CvMapGeneratorUtil
from CvMapGeneratorUtil import FractalWorld
from CvMapGeneratorUtil import TerrainGenerator
from CvMapGeneratorUtil import FeatureGenerator

#I don't know where this shows up!  You would think this would be
#used as the text for the Map: button, but that uses the name of
#the script instead!
def getDescription():
	return unicode("Fair Continents")

#this map has 2 custom options: number of continents horizontal and vertical
def getNumCustomMapOptions():
	return 2
	
#horizontal and vertical option titles for the setup screen
#these get used to the left of the drop down selectors
def getCustomMapOptionName(argsList):
	print "getCustomMapOptionName",argsList
	index = argsList[0]
	translated_text = [unicode("Horizontal:"),
	                   unicode("Vertical:")]
	return translated_text[index]
	
#there are 6 possible values for each: this tells the script
#how many entries the drop downs need to have
def getNumCustomMapOptionValues(argsList):
	print "getNumCustomMapOptionValues",argsList
	index = argsList[0]
	typeCount = [6,6]
	return typeCount[index]
	
#here are the 6 actual options for each, from 1 to 6 wide/tall	
def getCustomMapOptionDescAt(argsList):
	print "getCustomMapOptionDescAt",argsList
	iOption = argsList[0]
	iSelection = argsList[1]
	selection_names = [
	                  ["1 wide",
	                   "2 wide",
	                   "3 wide",
	                   "4 wide",
	                   "5 wide",
	                   "6 wide"],
	                  ["1 tall",
	                   "2 tall",
	                   "3 tall",
	                   "4 tall",
	                   "5 tall",
	                   "6 tall"]]
	translated_text = unicode(selection_names[iOption][iSelection])
	return translated_text
	
#gets the default values for width, height, 0-based index
#into the list above, so if we're asking for index 0 (width default)
#then I return 2 (selects '3 wide' as the default)
#for the other index, I return 1 (2 tall default)
def getCustomMapOptionDefault(argsList):
	index = argsList[0]
	if (index == 0): #asking for wide default
		return 2 #0 based index means 3rd entry == 3 wide
	return 1 #otherwise, must be asking for tall default 0 based means 2nd entry '2 tall'

#this is an advanced map, only shows up if you do custom map
def isAdvancedMap():
	"This map should not show up in simple mode"
	return 1

#this map uses the 'sea level' option to determine how much ocean
#goes between the continents (return 0 to disable the sea level selector)
def isSeaLevelMap():
	return 1

#starts the player's warrior and settler on the same tile if true
def startHumansOnSameTile():
	return True
	
#Step 1: the map grid size is determined you map will contain 4 tiles
# for every grid size, so an 8x4 grid size is 32x16 tiles
#allows you to override the grid sizes if your map has some particular need
#in my case, i'm going to force all the grid sizes to be friendly to ocean
#stripe placement.  Most maps could omit this function.
def getGridSize(argsList):
	if (argsList[0] == -1): # (-1,) is passed to function on loads
		return []

	#in an unusual variation, we'll add to the gridsize based on how much ocean we
	#are required to place, to make sure it is always possible to have some land

	gc = CyGlobalContext()
	map = CyMap()
	# Sea Level adjustment (from user input), transformed into width of sea grids
	sea = gc.getSeaLevelInfo(map.getSeaLevel()).getSeaLevelChange()
	seaWidth = 2 #default width for ocean stripes for normal sea level
	if (sea > 0):
		seaWidth = 3 #high sea level means extra width on the ocean stripes
	if (sea < 0):
		seaWidth = 1 #low sea level mean little width on the ocean stripes
	print "seawidth",seaWidth

	numHorStripes = int(map.getCustomMapOption(0)) + 1
	numVerStripes = int(map.getCustomMapOption(1)) + 1
	
	#how much extra vertical and horizontal is needed
	# multiply the number of stripes by their width (the actual width is seaWidth doubled)
	# and finally divide by 4, which is the plot to tile ratio
	eH = int((numHorStripes * seaWidth * 2) / 4)
	print "eH",eH
	eV = int((numVerStripes * seaWidth * 2) / 4)
	print "eV",eV


	# Choose grid sizes friendly to ocean stripe placement, and also
	# 2*wide vs tall so that horizontal and vertical travel are fair.
	# the longest horizontal trip is 1/2 width due to being able to navigate
	# in either direction.  the longest vertical trip is 1 *height because
	# height doesn't wrap
	# Most map scripts would just return fixed sizes here like:
	# 		WorldSizeTypes.WORLDSIZE_DUEL:		(6,3),
	grid_sizes = {
		WorldSizeTypes.WORLDSIZE_DUEL:		(6+eH,3+eV),
		WorldSizeTypes.WORLDSIZE_TINY:		(8+eH,4+eV),
		WorldSizeTypes.WORLDSIZE_SMALL:		(12+eH,6+eV),
		WorldSizeTypes.WORLDSIZE_STANDARD:	(16+eH,8+eV),
		WorldSizeTypes.WORLDSIZE_LARGE:		(20+eH,10+eV),
		WorldSizeTypes.WORLDSIZE_HUGE:		(32+eH,16+eV)
	}

	[eWorldSize] = argsList
	return grid_sizes[eWorldSize]
	

#Step 2: lay out the plot types, which is a sort of broad view of the map tiles
#basically telling you if you are looking at ocean, land, hill, or mountain
def generatePlotTypes():
	NiTextOut("Setting Plot Types (Python Fair Continents) ...")
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	iW = map.getGridWidth()
	widthLimit = iW-1 #range of array is 0..iW-1
	iH = map.getGridHeight()
	heightLimit = iH-1 #range of array is 0..iH-1
	plotTypes = [PlotTypes.PLOT_OCEAN] * (iW*iH)

	# Sea Level adjustment (from user input), transformed into width of sea grids
	sea = gc.getSeaLevelInfo(map.getSeaLevel()).getSeaLevelChange()
	seaWidth = 2 #default width for ocean stripes for normal sea level
	if (sea > 0):
		seaWidth = 3 #high sea level means extra width on the ocean stripes
	if (sea < 0):
		seaWidth = 1 #low sea level mean little width on the ocean stripes
	print "seawidth",seaWidth

    #put one big blob of land in the center, surrounded by a seawidth worth of 
    #ocean on all sides
	for plotX in range(iW):
		for plotY in range(iH):
			index = (plotY*iW) + plotX
			if (plotY < seaWidth or plotY > heightLimit-seaWidth or 
				plotX < seaWidth or plotX > widthLimit - seaWidth):
				plotTypes[index] = PlotTypes.PLOT_OCEAN
			else:
				plotTypes[index] = PlotTypes.PLOT_LAND
				
	#now add horizontal stripes to separate the continents
	#first figure out how wide each land strip should be			
	
	#number of horizontal stripes, note this may be 0 stripes if we are only doing one continent
	numHorStripes = int(map.getCustomMapOption(0))
	print "numHorStripes",numHorStripes
	#the amount of total land avail is the height - the ocean border - the stripes
	vertLandAvail = iH - (2*seaWidth) - (2*seaWidth*numHorStripes)
	avgVertLand = int(vertLandAvail / (numHorStripes + 1))
	
	landUse = []
	#add average size blocks of land, numbering equal to expected continent count
	for landBlock in range(0,(numHorStripes+1)):
		landUse.append(avgVertLand)
		vertLandAvail -= avgVertLand
	#add any unused land space, preferring to add to outer continents
	# because inner continents have other advantages
	landUseLen = len(landUse)
	landBlockIndexes = [0,landUseLen-1,1,landUseLen-2,2,landUseLen-3,3,landUseLen-4]
	while (vertLandAvail > 0):
		for landIndex in landBlockIndexes:
			if (landIndex >= 0 and landIndex < landUseLen and vertLandAvail > 0):
				landUse[landIndex] += 1
				vertLandAvail -= 1
				
	for landSize in landUse:
		print "vertical land size", landSize

	landUse.pop() #remove the last block of land because we don't need to process after it

	#finally, actually place the horizontal stripes of ocean
	horIndex = seaWidth #first block of land starts indented by the sea border
	for landSize in landUse:
		horIndex += landSize #step forward over the block of land
		#now lay a stripe down at the position we just stepped forward to
		for ypos in range(horIndex, horIndex + (seaWidth*2)):
			print "ocean horizontal stripe at y", ypos
			for xpos in range(iW):
				index = (ypos*iW) + xpos
				plotTypes[index] = PlotTypes.PLOT_OCEAN
		#and finally step forward by the width of the stripe we just laid down
		horIndex += (seaWidth*2)
		
	#now add vertical stripes to separate the continents (same process as the horizontal)
	#first figure out how wide each land strip should be			
	
	#number of vertical stripes, note this may be 0 if we are only doing one continent
	numVerStripes = int(map.getCustomMapOption(1))
	print "numVerStripes",numVerStripes
	#the amount of total land avail is the width - the ocean border - the stripes
	horLandAvail = iW - (2*seaWidth) - (2*seaWidth*numVerStripes)
	avgHorLand = int(horLandAvail / (numVerStripes + 1))
	
	landUse = []
	#add average size blocks
	for landBlock in range(0,(numVerStripes+1)):
		landUse.append(avgHorLand)
		horLandAvail -= avgHorLand
	#add any unused land space, preferring to add to outer continents
	# because inner continents have other advantages
	landUseLen = len(landUse)
	landBlockIndexes = [0,landUseLen-1,1,landUseLen-2,2,landUseLen-3,3,landUseLen-4]
	while (horLandAvail > 0):
		for landIndex in landBlockIndexes:
			if (landIndex >= 0 and landIndex < landUseLen and horLandAvail > 0):
				landUse[landIndex] += 1
				horLandAvail -= 1
				
	for landSize in landUse:
		print "horizontal land size", landSize

	landUse.pop() #remove the last block of land because we don't need to process after it

	#finally, actually place the vertical stripes of ocean
	verIndex = seaWidth #first block of land starts indented by the sea border
	for landSize in landUse:
		verIndex += landSize #step forward over the block of land
		#now lay a stripe down at the position we just stepped forward to
		for xpos in range(verIndex, verIndex + (seaWidth*2)):
			print "oceanvertical stripe at x", xpos
			for ypos in range(iH):
				index = (ypos*iW) + xpos
				plotTypes[index] = PlotTypes.PLOT_OCEAN
		#and finally step forward by the width of the stripe we just laid down
		verIndex += (seaWidth*2)


	# Add some random ocean to unsquarify the continents
	#directions identify the eight neighbor tiles
	directions = ( (-1,-1), (-1,0), (0,-1), (-1,1), (1,-1), (1,0), (0,1), (1,1) )
	#amount of extra ocean to add is proportional to size of map, and reduced by 
	#a factor dependant on the sea level setting
	iExtraOcean = int((iW*iH) / (6-seaWidth))
	while iExtraOcean > 0:
		iExtraOcean = iExtraOcean - 1
		oceanX = 1 + dice.get(iW-3, "Random Ocean x - Fair Continents")
		oceanY = 1 + dice.get(iH-3, "Random Ocean y - Fair Continents")
		index = (oceanY*iW) + oceanX
		#check for a nearby ocean, this makes sure every added ocean touches the main ocean
		#note that we don't have to worry about going off the board because we've restricted
		#the search to positions at least one indent inside the board
		for direction in directions:
			xOffset, yOffset = direction
			nearIndex = ( ((oceanY+yOffset)*iW) + (oceanX + xOffset))
			if (plotTypes[nearIndex] == PlotTypes.PLOT_OCEAN):
				plotTypes[index] = PlotTypes.PLOT_OCEAN
				print "ocean touch extra x", oceanX, "y", oceanY

	# Add some random lakes to make terrain more interesting
	# adds about 4% lakes, more for higher sea level, and note also that many of these
	# may land on existing ocean, in which case they don't become lakes
	iExtraOcean = int((iW*iH) / (25 - seaWidth))
	while iExtraOcean > 0:
		iExtraOcean = iExtraOcean - 1
		oceanX = dice.get(iW, "Random Ocean x - Fair Continents")
		oceanY = dice.get(iH, "Random Ocean y - Fair Continents")
		index = (oceanY*iW) + oceanX
		print "lake x", oceanX, "y", oceanY
		plotTypes[index] = PlotTypes.PLOT_OCEAN

	# Add some random hills to make terrain more interesting
	#note this isn't quite 1/7 = 14% hills, because it only hills if the random
	#draw hits a land tile (any ocean hit just vanishes to prevent growing the continents
	# in an unfair way)
	iExtraHills = int((iW*iH) / 7)
	while iExtraHills > 0:
		iExtraHills = iExtraHills - 1
		hillX = dice.get(iW, "Random Hill x - Fair Continents")
		hillY = dice.get(iH, "Random Hill y - Fair Continents")
		index = (hillY*iW) + hillX
		print "hill x", hillX, "y", hillY
		if plotTypes[index] == PlotTypes.PLOT_LAND:
			plotTypes[index] = PlotTypes.PLOT_HILLS

	# Add some random peaks to make terrain more interesting
	#note that this isn't quite adding 1/15 = 6.6% peaks because it only peaks when
	#the random draw is near a hill or an existing peak
	iExtraPeaks = int((iW*iH) / 15)
	while iExtraPeaks > 0:
		iExtraPeaks = iExtraPeaks - 1
		peakX = 1+dice.get(iW-3, "Random Peak x - Fair Continents")
		peakY = 1+dice.get(iH-3, "Random Peak y - Fair Continents")
		index = (peakY*iW) + peakX
		#check for a nearby hill, this makes sure every added peak touches a hill
		#note that we don't have to worry about going off the board because we've restricted
		#the search to positions at least one indent inside the board
		for direction in directions:
			xOffset, yOffset = direction
			nearIndex = ( ((peakY+yOffset)*iW) + (peakX + xOffset))
			#the plot converted must be land or hill, and must touch an existing hill or peak
			#this is not a game requirement, but rather a 'nice appearance' requirement
			if ((plotTypes[index] == PlotTypes.PLOT_LAND or plotTypes[index] == PlotTypes.PLOT_HILLS) and 
				(plotTypes[nearIndex] == PlotTypes.PLOT_HILLS or plotTypes[nearIndex] == PlotTypes.PLOT_PEAK)):
				plotTypes[index] = PlotTypes.PLOT_PEAK
				print "peak x", peakX, "y", peakY

	# Finished generating plots!
	return plotTypes
	
#Step 3: now generate the fine terrain, in particular, need to decide if 
#any land tile is desert, plains, grass, tundra, or snow
def generateTerrainTypes():		
	NiTextOut("Generating Terrain (Fair Continents) ...")
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	terrainData = [TerrainTypes.NO_TERRAIN]*(iW*iH)
	
	terrainDesert = gc.getInfoTypeForString("TERRAIN_DESERT")
	terrainPlains = gc.getInfoTypeForString("TERRAIN_PLAINS")
	terrainGrass = gc.getInfoTypeForString("TERRAIN_GRASS")
	terrainTundra = gc.getInfoTypeForString("TERRAIN_TUNDRA")
	terrainSnow = gc.getInfoTypeForString("TERRAIN_SNOW")
	midmap = int(iH / 2)
	desertRange = int(iH / 8)
	mindesert = int(midmap - desertRange)
	maxdesert = int(midmap + desertRange)
	
	#for every terrain we need to consider in x,y
	for x in range(iW):
		for y in range(iH):
			pPlot = map.plot(x,y)
			index = (y*iW) + x
			plotType = pPlot.getPlotType()
			plotRand = dice.get(100, "Random Terrain - Fair Continents Python")
			#water tiles, peaks, and hills should always take their default terrain type
			if (pPlot.isWater()):
				terrainData[index] = pPlot.getTerrainType()
			elif (plotType == PlotTypes.PLOT_PEAK):	
				terrainData[index] = pPlot.getTerrainType()
			elif (plotType == PlotTypes.PLOT_HILLS):	
				terrainData[index] = pPlot.getTerrainType()
			#land tiles actually have choices in what type of land tile to be
			elif (plotType == PlotTypes.PLOT_LAND):
				#note that iH is at the top of the map (northmost), when you might expect bottom
				#this allows 10% snow only near the very furthest north or south strip of land
				if ( (y < 4 or y > iH-5) and plotRand < 10):
					terrainData[index] = terrainSnow
				#this allows 50% tundra only near the north/south most land position
				elif ( (y < 6 or y > iH-7) and plotRand < 50):
					terrainData[index] = terrainTundra
				#50% grass
				elif (plotRand < 50):
					terrainData[index] = terrainGrass
				#40% plains
				elif (plotRand < 90):
					terrainData[index] = terrainPlains
				# 10% desert only allowed within 10% stripe in center of map
				elif ( (y > mindesert) and (y < maxdesert) ):
					terrainData[index] = terrainDesert
				#anything that didn't get decided otherwise is grass
				else:
					terrainData[index] = terrainGrass
			#this should never happen: what plot type could this be?  but just in case...
			else:
				terrainData[index] = pPlot.getTerrainType()
			#this should be impossible, but just in case, make sure every terrain is assigned
			if (terrainData[index] == TerrainTypes.NO_TERRAIN):
				terrainData[index] = pPlot.getTerrainType()
				
	return terrainData

#Step 4: add features to your terrain	
#this is responsible for things like oasis, ice floes on the ocean, rivers, goodies ...
def addFeatures():
	NiTextOut("Adding Features (Python Fair Continents) ...")
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	players = gc.getGame().countCivPlayersEverAlive()
	impGoodyHut = gc.getInfoTypeForString("IMPROVEMENT_GOODY_HUT")
	dirN = gc.getInfoTypeForString("CARDINALDIRECTION_NORTH")
	dirE = gc.getInfoTypeForString("CARDINALDIRECTION_EAST")
	dirS = gc.getInfoTypeForString("CARDINALDIRECTION_SOUTH")
	dirW = gc.getInfoTypeForString("CARDINALDIRECTION_WEST")
	featIce = gc.getInfoTypeForString("FEATURE_ICE")
	featJungle = gc.getInfoTypeForString("FEATURE_JUNGLE")
	featOasis = gc.getInfoTypeForString("FEATURE_OASIS")
	featFlood = gc.getInfoTypeForString("FEATURE_FLOOD_PLAINS")
	featForest = gc.getInfoTypeForString("FEATURE_FOREST")
	allDirs = (dirN, dirE, dirS, dirW)
	allDirTypes = [CardinalDirectionTypes.CARDINALDIRECTION_NORTH,
					CardinalDirectionTypes.CARDINALDIRECTION_EAST,
					CardinalDirectionTypes.CARDINALDIRECTION_SOUTH,
					CardinalDirectionTypes.CARDINALDIRECTION_WEST]
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	midLine = int(iH / 2)

	# Sea Level adjustment (from user input), transformed into width of sea grids
	sea = gc.getSeaLevelInfo(map.getSeaLevel()).getSeaLevelChange()
	seaWidth = 2 #default width for ocean stripes for normal sea level
	if (sea > 0):
		seaWidth = 3 #high sea level means extra width on the ocean stripes
	if (sea < 0):
		seaWidth = 1 #low sea level mean little width on the ocean stripes
	print "seaWidth",seaWidth
	
	#first pass: place rivers.  if you don't place rivers first
	#then oasis and floodplains are impossible
	#2% of hills or peaks spawns a river in a random direction			
	for x in range(iW):
		for y in range(iH):
			pPlot = map.plot(x,y)
			riverOdds = dice.get(100, "River Placement Odds - Fair Continents PYTHON")
			if pPlot.isHills() or pPlot.isPeak() and (riverOdds < 2):
				index = dice.get(len(allDirs), "River Placement - Fair Continents PYTHON")
				CyMapGenerator().doRiver(pPlot, allDirTypes[index])

	
	for x in range(iW):
		for y in range(iH):
			pPlot = map.plot(x,y)
			percRangeFromEquator = int (((midLine - y) * 100) / midLine)
			yDiffFromBorder = y
			if y > midLine:
				percRangeFromEquator = int (((y - midLine) * 100) / midLine)
				yDiffFromBorder = iH-y
			featureOdds = dice.get(100, "Feature Placement Odds - Fair Continents PYTHON")
			#ice
			if (y<=0 or y>=iH-1) and pPlot.canHaveFeature(featIce):
				pPlot.setFeatureType(featIce,-1)
			elif yDiffFromBorder<=(seaWidth+3) and pPlot.canHaveFeature(featIce) and featureOdds < 12*(3+seaWidth-yDiffFromBorder):
				pPlot.setFeatureType(featIce,-1)
				
			#oasis don't ask if it canHaveFeature oasis because that always returns false
			elif pPlot.isWater() == false and pPlot.isFlatlands() and pPlot.isFreshWater() and featureOdds<2:
				pPlot.setFeatureType(featOasis,-1)

			#flood same issue as oasis, always false for some reason
			elif pPlot.isFreshWater() and featureOdds<5:
				pPlot.setFeatureType(featFlood,-1)

			#jungle
			elif percRangeFromEquator < 50 and pPlot.canHaveFeature(featJungle) and featureOdds< int( (100-percRangeFromEquator) / 2):
				pPlot.setFeatureType(featJungle,-1)
	
	
			#forest
			elif percRangeFromEquator > 25 and pPlot.canHaveFeature(featForest)	and featureOdds< int( (percRangeFromEquator) / 2):
				pPlot.setFeatureType(featForest,-1)


	plotsForGoodies = []
	for plotIndex in range(iW * iH):
		pPlot = map.plotByIndex(plotIndex)
		# A general map post processing example: remove all peaks along the ocean
		if pPlot.isPeak() and pPlot.isCoastalLand():
			pPlot.setPlotType(PlotTypes.PLOT_HILLS, true, true)
		#record all plots that are potential goody locations
		if pPlot.canHaveImprovement(impGoodyHut, TeamTypes.NO_TEAM, false):
			plotsForGoodies.append(pPlot)
			

	#add randomly placed goody huts, 8 per player overall
	goodyHutCount = 8 * players
	while (goodyHutCount > 0):
		goodyHutCount -= 1
		if len(plotsForGoodies) <= 0:
			break
		index = dice.get(len(plotsForGoodies), "GoodyHut Placement - Fair Continents PYTHON")
		pPlot = plotsForGoodies[index]
		pPlot.setImprovementType(impGoodyHut)
		del plotsForGoodies[index]
		
	return 0

#handled in addfeatures, so unnecessary
def normalizeRemovePeaks():
	return None

#Step 5: add bonuses to your map
#this method puts down the tile bonuses, such as whale, rice, oil, etc.	
def addBonusType(argsList):
	[iBonusType] = argsList
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	type_string = gc.getBonusInfo(iBonusType).getType()

	#these are the resources you find in water tiles
	resourcesWater = ('BONUS_CLAM', 'BONUS_FISH', 'BONUS_WHALE', 'BONUS_CRAB')
	#these resources should be given more often than others
	resourcesCommon = ('BONUS_STONE', 'BONUS_MARBLE', 'BONUS_BANANA', 'BONUS_CORN', 
	'BONUS_COW', 'BONUS_DEER', 'BONUS_PIG', 'BONUS_RICE', 
	'BONUS_SHEEP', 'BONUS_WHEAT')

	sizekey = map.getWorldSize()
	sizevalues = {
		WorldSizeTypes.WORLDSIZE_DUEL:		1,
		WorldSizeTypes.WORLDSIZE_TINY:		2,
		WorldSizeTypes.WORLDSIZE_SMALL:		4,
		WorldSizeTypes.WORLDSIZE_STANDARD:	7,
		WorldSizeTypes.WORLDSIZE_LARGE:		11,
		WorldSizeTypes.WORLDSIZE_HUGE:		16
		}
	sizemodifier = sizevalues[sizekey]
	#find out how many players in game, and make sure there are some for each
	players = gc.getGame().countCivPlayersEverAlive()
	playerExtra = dice.get(int(players / 2), "Extra resources per player - Fair Continents Python")
	extraFactor = 0
	
	#the bountiful ocean - I add extra resources to water tiles because there
	#is a lot of coastal land in this map design
	if (type_string in resourcesWater):
		extraFactor += int((5*players)/3); #extra water bonuses
		
	#some resources are meant to be a bit more common
	if (type_string in resourcesCommon):
		extraFactor += int((4*players)/3); #extra common bonuses
	
	#total is based on: size of map, num players, random extra per player, extra for water types
	count = int(sizemodifier + playerExtra + extraFactor)
	
	# Set plot eligibility for current bonus.
	# Begin by initiating the list, into which eligible plots will be recorded.
	eligible = []
	for x in range(iW):
		for y in range(iH):
			# First check the plot for an existing bonus.
			pPlot = map.plot(x,y)
			if pPlot.getBonusType(-1) != -1:
				continue # to next plot.
			if pPlot.getFeatureType() == gc.getInfoTypeForString("FEATURE_OASIS"):
				continue # Soren wants no bonuses in oasis plots. So mote it be.
			# Check plot type and features for eligibility.
			if (pPlot.canHaveBonus(iBonusType, True)):
				eligible.append([x,y])
                                    
	# Now we assign the bonuses to eligible plots chosen completely at random
	# from among the valid plots found above
	while count > 0:
		if eligible == []: break # No eligible plots left!
		index = dice.get(len(eligible), "Bonus Placement - Fair Continents PYTHON")
		[x,y] = eligible[index]
		map.plot(x,y).setBonusType(iBonusType)
		del eligible[index] # Remove this plot from the eligible list.
		count = count - 1  # Reduce number of bonuses left to place.
	# This bonus type is done.
	
	# Uncomment to Let C also handle this bonus in the default way.
	#CyPythonMgr().allowDefaultImpl() 

#Step 6: place all the players
#this method is still to come