/*****************************************************************************/
// tcDecoder.cpp
// Douglas Kane 5 OCT 2001
// foster_fc@yahoo.com
// This file is released to the public domain without restriction. I'd
// appreciate credit if this sees any use. Please post or e-mail me code
// updates for any improvements that are made by others.
//
// tcDecoder class code -- tcDecoder implements PKWARE Data Compression Library
// decompression as released by Ben Rudiak-Gould (jcxz@hotmail.com) on
// comp.compression newsgroup. Prefix-coded literals aren't implemented. I
// have not tested this with large files. There may be problems with the circ
// dictionary implementation.
/*****************************************************************************/

#include "tcDecoder.h"
#include <stdlib.h>

/*----------------------------------------------------------------------------*/
// The default constructor isn't supported.
tcDecoder::tcDecoder() {
}
/*----------------------------------------------------------------------------*/
// Use this constructor.
tcDecoder::tcDecoder(tcBitstream *apBitstream) {
   int lnBits;

   mpBitstream = apBitstream;
   
   lnBits = mpBitstream->Get(8);
   printf("Literal setting: %d\n",lnBits);

   if (lnBits == 1) {
      //printf("Error: Variable-length literals are not supported.\n");
      exit(2);
   }
   else if (lnBits != 0) {
      //printf("Error: Bad header.\n");
      exit(3);
   }
   
   lnBits = mpBitstream->Get(8);
   mnSizeByte = (unsigned char)lnBits;
   switch (lnBits) {
   case 4:
      mnDictionarySize = 1024;
      break;
   case 5:
      mnDictionarySize = 2048;
      break;
   case 6:
      mnDictionarySize = 4096;
	  //printf("Size: 4096.\n");
      break;
   default:
      //printf("Error: Bad second header byte.\n");
      exit(4);
   }

   mnCurrentPos = 0;
}
/*----------------------------------------------------------------------------*/
tcDecoder::~tcDecoder() {
}
/*----------------------------------------------------------------------------*/
void tcDecoder::Decode(FILE *lpOutFile) {
   unsigned int lnIndex, lnStartIndex, lnCopyCount;
   unsigned int lnNextPos;
   tsToken lsToken;

   //*apSize = 0;
   lsToken.meType = LITERAL; // initialize to enter loop properly 

   while ((lsToken.meType == LITERAL)||(lsToken.meType ==LENGTH_OFFSET)) {
      GetNextToken(&lsToken);
      if(lsToken.meType == LITERAL) {
       //  apBuffer[(*apSize)++] = lsToken.mnLiteral;
		 fputc(lsToken.mnLiteral,lpOutFile);
         maDictionary[mnCurrentPos++] = lsToken.mnLiteral;
         if (mnCurrentPos == mnDictionarySize) mnCurrentPos = 0;
     //    if (*apSize >= anBufferSize) return;
      }
      else if (lsToken.meType == LENGTH_OFFSET) {
         lnStartIndex = (mnCurrentPos-1-lsToken.mnOffset) % mnDictionarySize;
         lnIndex = lnStartIndex;
         lnNextPos = mnCurrentPos;
         lnCopyCount = 0;

         while(lnCopyCount++ < lsToken.mnLength) {
            //apBuffer[(*apSize)++] = maDictionary[lnIndex];
			fputc(maDictionary[lnIndex],lpOutFile);
            maDictionary[lnNextPos++] = maDictionary[lnIndex++];
            if (lnIndex == mnCurrentPos) lnIndex = lnStartIndex;
			if (lnIndex == mnDictionarySize) lnIndex = 0;
            if (lnNextPos == mnDictionarySize) lnNextPos = 0;
            //if (*apSize >= anBufferSize) return;
         }
         mnCurrentPos = lnNextPos;
      }
      else if (lsToken.meType == END_OF_STREAM) {}
      else if (lsToken.meType == ERROR) {}
   } // while
}


/*----------------------------------------------------------------------------*/
void tcDecoder::GetNextToken(tsToken *apToken) {
   int lnBits;
   int lnLowOrderBits;

   lnBits = mpBitstream->Get(1);
   if (lnBits < 0) {
      apToken->meType = ERROR;
   }
   else if (lnBits == 0) {
      apToken->meType = LITERAL;
      apToken->mnLiteral = mpBitstream->Get(8);
   }
   else {
      apToken->meType = LENGTH_OFFSET;
      apToken->mnLength = DecodeCopyLength();
      if (apToken->mnLength == 519) apToken->meType = END_OF_STREAM;
      else {
         lnLowOrderBits = (apToken->mnLength == 2) ? 2 : mnSizeByte;
         apToken->mnOffset = DecodeCopyOffset(lnLowOrderBits); 
      }
   }
}
/*----------------------------------------------------------------------------*/
/* DecodeCopyLength() uses the prefix code in this table:
 2        101
 3        11
 4        100
 5        011
 6        0101
 7        0100
 8        0011
 9        00101
 10-11    00100x
 12-15    00011xx
 16-23    00010xxx
 24-39    000011xxxx
 40-71    000010xxxxx
 72-135   000001xxxxxx
 136-263  0000001xxxxxxx
 264-518  0000000xxxxxxxx

 This version is somewhat prone to error and is slow. LOL, there has
to be an
 easier way to do this, e.g. based on a formula for creating the tree.
*/
int tcDecoder::DecodeCopyLength(void) {
   int lnBits;

   lnBits = mpBitstream->Get(2);

   if (lnBits == 0x3) return 3; // 11b -> 3
   if (lnBits == 0x1) { // 10b
      lnBits = mpBitstream->Get(1);
      if (lnBits == 0x1) return 2; // 101b -> 2
      else return 4; // 100b -> 4
   }
   else if (lnBits == 0x2) { // 01b
      if (mpBitstream->Get(1)) return 5; // 011b -> 5
      else { // 010b
         if (mpBitstream->Get(1)) return 6; // 0101b -> 6
         else return 7; // 0100b -> 7
      }
   }
   else { // 00b
      if (mpBitstream->Get(1)) { // 001b
         if (mpBitstream->Get(1)) return 8; // 0011b -> 8
         else { // 0010b
            if (mpBitstream->Get(1)) return 9; // 00101b -> 9
            else return (10 + mpBitstream->Get(1)); // 00100xb -> 10-11
         }
      }
      else { // 000b
         if (mpBitstream->Get(1)) { // 0001b
            if (mpBitstream->Get(1)) {
               return (12 + mpBitstream->Get(2)); // 00011xxb -> 12-15
            }
            else {
               return (16 + mpBitstream->Get(3)); // 00010xxxb -> 16-23
            }
         }
         else { // 0000b
            lnBits = mpBitstream->Get(2);
            if (lnBits == 0x3) 
               return (24 + mpBitstream->Get(4)); // 000011xxxxb -> 24-39
            else if (lnBits == 0x1) 
               return (40 + mpBitstream->Get(5)); // 000010xxxxxb -> 40-71
            else if (lnBits == 0x2) 
               return (72 + mpBitstream->Get(6)); // 000001xxxxxxb -> 72-135
            else { // 000000b
               if (mpBitstream->Get(1)) 
                  return (136 + mpBitstream->Get(7)); //0000001xxxxxxxb -> 136-263
               else 
                  return (264 + mpBitstream->Get(8)); //0000000xxxxxxxxb -> 264-518, 519
            } // 000000b
         } // 0000b
      } // 000b
   } // 00b

   return -1;
}
/*----------------------------------------------------------------------------*/
/* DecodeCopyOffset(int anLowOrderBits) uses the prefix code in this
table:
 00   11          20   0010111
 01   1011        21   0010110
 02   1010        22   0010101
 03   10011       23   0010100
 04   10010       24   0010011
 05   10001       25   0010010
 06   10000       26   0010001
 07   011111      27   0010000
 08   011110      28   0001111
 09   011101      29   0001110
 0a   011100      2a   0001101
 0b   011011      2b   0001100
 0c   011010      2c   0001011
 0d   011001      2d   0001010
 0e   011000      2e   0001001
 0f   010111      2f   0001000
 10   010110      30   00001111
 11   010101      31   00001110
 12   010100      32   00001101
 13   010011      33   00001100
 14   010010      34   00001011
 15   010001      35   00001010
 16   0100001     36   00001001
 17   0100000     37   00001000
 18   0011111     38   00000111
 19   0011110     39   00000110
 1a   0011101     3a   00000101
 1b   0011100     3b   00000100
 1c   0011011     3c   00000011
 1d   0011010     3d   00000010
 1e   0011001     3e   00000001
 1f   0011000     3f   00000000

 anLowOrderBits should be 4, 5, or 6 based on dictionary length
 except if the copy length is 2 when it should be 2. 
*/
int tcDecoder::DecodeCopyOffset(unsigned int anLowOrderBits) {
   int lnBits;

   lnBits = mpBitstream->Get(2);
   if (lnBits == 0x3) // 11b 
      return CalcOffset(0x00, anLowOrderBits); // 11b -> 0x00
   else if (lnBits == 0x1) { // 10b
      if(mpBitstream->Get(1)==1) { // 101b
         if(mpBitstream->Get(1)==1) // 1011b
            return CalcOffset(0x01, anLowOrderBits); // 0x01
         else // 1010b
            return CalcOffset(0x02, anLowOrderBits); // 0x02
      }
      else { // 100b
         lnBits = mpBitstream->Get(2);
         return CalcOffset(0x06 - BitReverse(lnBits,2),anLowOrderBits); // 0x03 - 0x06
      }
   }
   else if (lnBits == 0x2) { // 01b
      lnBits = mpBitstream->Get(4);
      if (lnBits != 0x0) 
         return CalcOffset(0x16 - BitReverse(lnBits,4),anLowOrderBits); // 0x07 - 0x15
      else { // 010000b
         return CalcOffset(0x17 - mpBitstream->Get(1),anLowOrderBits); // 0x16 - 0x17
      }
   }
   else { // 00b
      if (mpBitstream->Get(1)) // 001b
         return CalcOffset(0x27 - BitReverse(mpBitstream->Get(4),4), anLowOrderBits); // 0x18 - 0x27
      else { // 000b
         if (mpBitstream->Get(1)) // 0001b
            return CalcOffset(0x2F -BitReverse(mpBitstream->Get(3),3), anLowOrderBits); // 0x28 - 0x2F
         else // 0000b
            return CalcOffset(0x3F -BitReverse(mpBitstream->Get(4),4), anLowOrderBits); // 0x30 - 0x3F
      } // 000b

   } // 00b return -1;
}
/*----------------------------------------------------------------------------*/
// anHighByte is shifted and added to the next anLowOrderBits from the stream */
int tcDecoder::CalcOffset(unsigned char anHighByte, 
                          unsigned int anLowOrderBits) {
   return ((anHighByte << anLowOrderBits) |
mpBitstream->Get(anLowOrderBits));
}


