/*
 *
 *    Program to patch pediaicons.txt and civilopedia.txt given two files with a list of additions
 *    Other fields in those two files should not be modified at all
 *    Program assumes the files are found in C:\program files\Infogrames\Civilization III\Text
 *    If it doesnt find them there though it will also try ..\text and .\
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// All strings are forced to be this length or shorter
#define MAX_STRING_LENGTH 4096
#define TYPE_ICON   0   // For new_change to know which list it goes in
#define TYPE_PEDIA  1

// Structure to hold a single change field = "#RACE_WHATEVER"  value = "scenarios\mod\leader.pcx" for example
// Otherwise its a simple linked list terminated by NULL
typedef struct change CHANGE;
struct change
{
	char field[MAX_STRING_LENGTH];
	char value[MAX_STRING_LENGTH];
	bool found;  // Marked true if the field was found in the file when scanned
	CHANGE * next;
};

// The two types of changes
CHANGE * icon_changes = NULL;
CHANGE * pedia_changes = NULL;

// Function prototypes documentation with functions
char * fread_to_eol(char * buf, FILE * file);
void new_change(int type, char * field, char * value);
void load_changes_icons(char * filename);
bool scan_for_changes(char * filename, int type);
bool apply_changes(char * filename, int type);
char * lower_case(char * buf, char * original);

int main(int n, char * args[])
{
	char filename[MAX_STRING_LENGTH];
	bool found = false;

	if (n >= 2)
		sprintf(filename, "%s", args[1]);

	// Load our set of changes
	load_changes_icons(filename);

	if (scan_for_changes("pediaicons.txt", TYPE_ICON))
	{
		found = true;
		apply_changes("pediaicons.txt", TYPE_ICON);
	}
	// If it wasnt found yet, try the full path the double slashes are just single slashes after \n type codes process
	if (found == false && scan_for_changes("C:\\Program Files\\Infogrames Interactive\\Civilization III\\Text\\pediaicons.txt", TYPE_ICON))
	{
		found = true;
		apply_changes("C:\\Program Files\\Infogrames Interactive\\Civilization III\\Text\\pediaicons.txt", TYPE_ICON);
	}
	// If its still not found make one last attempt to find it by ..\text
	if (found == false && scan_for_changes("..\\text\\pediaicons.txt", TYPE_ICON))
	{
		found = true;
		apply_changes(",,\\text\\pediaicons.txt", TYPE_ICON);
	}

	if (!found)
	{
		printf("Error:  pediaicons.txt not found.\n");
		exit( -1);
	}

	printf("Patch applied to pediaicons.txt");
	return 0;
}

// Load a set of changes from filename for pediaicons.txt
void load_changes_icons(char * filename)
{
	FILE * file;
	char field[MAX_STRING_LENGTH];
	char value[MAX_STRING_LENGTH];
	char buf[MAX_STRING_LENGTH]; 
	char *tmp;
	char alpha;

	file = fopen(filename, "r");
	if (file == NULL)
	{
		printf("Patch file not found.\n");
		exit(-1);
	}

	while (!feof(file))
	{
		tmp = fread_to_eol(buf, file);
		sprintf(field, "%s", tmp);
		
		value[0] = '\0';  // Be sure its empty
		alpha = '\0';  // So its not a # obviously :)
		while (!feof(file) && alpha != '#')
		{
			alpha = (char) fgetc(file);
			if (!feof(file) && alpha != '#')
			{
				tmp = fread_to_eol(buf+1, file);  // Messy but I do +1, so I can just append alpha easily
				if (strlen(value) + strlen(buf) + 2 < MAX_STRING_LENGTH)
				{
					buf[0] = alpha;
					strcat(value, buf);
					strcat(value, "\n");
				}
			}
		}
		if (alpha == '#')
			ungetc('#', file);  // So its there for our next round of getting a change

		if (field[0] == '#')
			new_change(TYPE_ICON, field, value);  // Create a new item to be modified
	}

	fclose(file);
}

// Function simply reads to the end of a line and returns the result
char * fread_to_eol(char * buf, FILE * file)
{
	char alpha;
	int cnt = 0;

	while (!feof(file))
	{
		alpha = (char) fgetc(file);
		if (alpha == '\n' || alpha == '\r') break;
		if (cnt < MAX_STRING_LENGTH -1) // Enforce our string length rules
		{
			buf[cnt] = alpha;
			cnt++;
		}
	}
	buf[cnt] = '\0';
	return buf;
}

// A quick utility function to lowercase an entire string and return a pointer
// Allows us to eliminate case sensitivity in string compares
char * lower_case(char * buf, char * original)
{
	int cnt;
	
	cnt = 0;
	while (original[cnt] != '\0')
	{
		buf[cnt] = tolower(original[cnt]);
		cnt++;
	}
	buf[cnt] = '\0';

	return buf;
}

// Create a new change item with the field equal to buf
void new_change(int type, char * field, char * value)
{
	CHANGE * change = (CHANGE *) malloc(sizeof(CHANGE));
	
	// Link this change
	if (type == TYPE_ICON)
	{
		change->next = icon_changes;
		icon_changes = change;
	}
	if (type == TYPE_PEDIA)
	{
		change->next = pedia_changes;
		pedia_changes = change;
	}
	// error checking
	if (strlen(field) >= MAX_STRING_LENGTH) field[MAX_STRING_LENGTH-1] = '\0';
	if (strlen(value) >= MAX_STRING_LENGTH) value[MAX_STRING_LENGTH-1] = '\0';

	sprintf(change->field, "%s", field);
	sprintf(change->value, "%s", value);
	change->found = false;
	return;
}

// Scan a file for the presense of the change fields denoted by type
bool scan_for_changes(char *filename, int type)
{
	FILE * file;
	char buf[MAX_STRING_LENGTH];
	char buf2[MAX_STRING_LENGTH]; // To lowercase the change->field
	char buf3[MAX_STRING_LENGTH]; // To lowercase the original buf
	char * tmp;
	CHANGE * change;
	CHANGE * top_change;

	if (type == TYPE_ICON)
		top_change = icon_changes;
	if (type == TYPE_PEDIA)
		top_change = pedia_changes;

	file = fopen(filename, "r");

	if (file == NULL) return false;
	
	while (!feof(file))
	{
		tmp = fread_to_eol(buf, file);
		if (buf[0] == '#' && !feof(file))
		{
			change = top_change;
			while (change != NULL)
			{
				// If its the same field name mark it as found
				if (!strcmp(lower_case(buf2,change->field), lower_case(buf3, buf)))
					change->found = true;
				change = change->next;
			}
		}
	}

	fclose(file);
	return true;
}

// Go through and add any missing (not found) change fields of the type asked to the file asked
bool apply_changes(char * filename, int type)
{
	FILE * file;
	CHANGE * change;
	CHANGE * top_change;
	bool found = false;

	if (type == TYPE_ICON)
		top_change = icon_changes;
	if (type == TYPE_PEDIA)
		top_change = pedia_changes;

	file = fopen(filename, "a");  // Append operation
	if (file == NULL) return false;  // Not necessarily an error

	fprintf(file, "\n");  // To space out what we added a bit so it stands out more
	// Go through our changes any missing one needs adding
	change = top_change;
	while (change != NULL)
	{
		if (change->found == false)
		{
			found = true;
			fprintf(file, "%s\n", change->field);
			fprintf(file, "%s", change->value);  // Already has a return
		}
		change = change->next;
	}
	
	fclose(file);

	if (!found)
		printf("No relevant changes found in PediaIcons.txt");
	return true;
}